<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Api\Point\Model;

use InPost\International\Country;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class Address implements \Stringable
{
    /**
     * @var string
     */
    private $country;

    /**
     * @var string|null
     */
    private $administrativeArea;

    /**
     * @var string
     */
    private $city;

    /**
     * @var string|null
     */
    private $street;

    /**
     * @var string|null
     */
    private $buildingNumber;

    /**
     * @var string
     */
    private $postalCode;

    public function __construct(string $country, string $city, string $postalCode, string $administrativeArea = null, string $street = null, string $buildingNumber = null)
    {
        $this->country = $country;
        $this->administrativeArea = $administrativeArea;
        $this->city = $city;
        $this->street = $street;
        $this->buildingNumber = $buildingNumber;
        $this->postalCode = $postalCode;
    }

    public function getCountry(): string
    {
        return $this->country;
    }

    public function getAdministrativeArea(): ?string
    {
        return $this->administrativeArea;
    }

    public function getCity(): string
    {
        return $this->city;
    }

    public function getStreet(): ?string
    {
        return $this->street;
    }

    public function getBuildingNumber(): ?string
    {
        return $this->buildingNumber;
    }

    public function getPostalCode(): string
    {
        return $this->postalCode;
    }

    public function __toString(): string
    {
        $country = Country::tryFrom($this->country) ?? Country::Poland();
        $streetAddress = $country->formatStreetAddress((string) $this->street, (string) $this->buildingNumber);

        if ('' === $streetAddress) {
            return sprintf('%s %s', $this->postalCode, $this->city);
        }

        return sprintf('%s, %s %s', $streetAddress, $this->postalCode, $this->city);
    }
}
