<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Api\Point\Request;

use InPost\International\Api\Point\Model\PointCapability;
use InPost\International\Country;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class PointsRequest
{
    /**
     * @var int|null
     */
    private $page;

    /**
     * @var int|null
     */
    private $perPage;

    /**
     * @var int|null
     */
    private $limit;

    /**
     * @var Country|null
     */
    private $country;

    /**
     * @var string|null
     */
    private $administrativeArea;

    /**
     * @var string|null
     */
    private $city;

    /**
     * @var string|null
     */
    private $postalCode;

    /**
     * @var PointCapability[]
     */
    private $capabilities;

    /**
     * @param PointCapability[] $capabilities
     */
    public function __construct(int $page = null, int $perPage = null, int $limit = null, Country $country = null, string $administrativeArea = null, string $city = null, string $postalCode = null, array $capabilities = [])
    {
        $this->page = $page;
        $this->perPage = $perPage;
        $this->limit = $limit;
        $this->country = $country;
        $this->administrativeArea = $administrativeArea;
        $this->city = $city;
        $this->postalCode = $postalCode;
        $this->capabilities = $capabilities;
    }

    public function getPage(): ?int
    {
        return $this->page;
    }

    public function getPerPage(): ?int
    {
        return $this->perPage;
    }

    public function getLimit(): ?int
    {
        return $this->limit;
    }

    public function getCountry(): ?Country
    {
        return $this->country;
    }

    public function getAdministrativeArea(): ?string
    {
        return $this->administrativeArea;
    }

    public function getCity(): ?string
    {
        return $this->city;
    }

    public function getPostalCode(): ?string
    {
        return $this->postalCode;
    }

    /**
     * @return PointCapability[]
     */
    public function getCapabilities(): array
    {
        return $this->capabilities;
    }

    public function toQueryParameters(): array
    {
        $params = [];

        if (null !== $this->page) {
            $params['page'] = (string) $this->page;
        }

        if (null !== $this->perPage) {
            $params['perPage'] = (string) $this->perPage;
        }

        if (null !== $this->limit) {
            $params['limit'] = (string) $this->limit;
        }

        if (null !== $this->country) {
            $params['address.country'] = $this->country->value;
        }

        if (null !== $this->administrativeArea) {
            $params['address.administrativeArea'] = $this->administrativeArea;
        }

        if (null !== $this->city) {
            $params['address.city'] = $this->city;
        }

        if (null !== $this->postalCode) {
            $params['address.postalCode'] = $this->postalCode;
        }

        if ([] !== $this->capabilities) {
            $params['capabilities'] = implode(',', array_map(static function (PointCapability $capability): string {
                return $capability->value;
            }, $this->capabilities));
        }

        return $params;
    }
}
