<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Api\Shipment\Model\Service;

use InPost\International\Country;
use InPost\International\Enum\StringEnum;

if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * @method static self Euro()
 * @method static self PolishZloty()
 */
final class Currency extends StringEnum
{
    private const EURO = 'EUR';
    private const POLISH_ZLOTY = 'PLN'; // not documented as a possible enum value, but is actually required to create a domestic shipment with insurance

    public static function getDefault(): self
    {
        return self::Euro();
    }

    public function convertToSmallestUnit(float $amount): int
    {
        switch ($this) {
            case self::Euro():
            case self::PolishZloty():
                return (int) ($amount * 100);
            default:
                throw new \LogicException('Not implemented.');
        }
    }

    /**
     * @return Country[]
     */
    public function getCountries(): array
    {
        switch ($this) {
            case self::PolishZloty():
                return [Country::Poland()];
            case self::Euro():
                return array_filter(Country::cases(), static function (Country $country): bool {
                    return $country !== Country::Poland();
                });
            default:
                throw new \LogicException('Not implemented.');
        }
    }
}
