<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Api\Shipment\Request;

use InPost\International\Api\Shipment\Model\Destination\DestinationAddress;
use InPost\International\Api\Shipment\Model\Origin\OriginPoint;
use InPost\International\Api\Shipment\Model\Parcel;
use InPost\International\Api\Shipment\Model\Recipient;
use InPost\International\Api\Shipment\Model\References;
use InPost\International\Api\Shipment\Model\Sender;
use InPost\International\Api\Shipment\Model\ShipmentPriority;
use InPost\International\Api\Shipment\Model\ValueAddedServices;
use InPost\International\Api\Shipment\ShipmentType;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class PointToAddressShipment implements ShipmentInterface
{
    /**
     * @var Sender
     */
    private $sender;

    /**
     * @var Recipient
     */
    private $recipient;

    /**
     * @var OriginPoint
     */
    private $origin;

    /**
     * @var DestinationAddress
     */
    private $destination;

    /**
     * @var ShipmentPriority
     */
    private $priority;

    /**
     * @var ValueAddedServices|null
     */
    private $valueAddedServices;

    /**
     * @var References|null
     */
    private $references;

    /**
     * @var Parcel
     */
    private $parcel;

    public function __construct(Sender $sender, Recipient $recipient, OriginPoint $origin, DestinationAddress $destination, ShipmentPriority $priority, Parcel $parcel, ValueAddedServices $valueAddedServices = null, References $references = null)
    {
        $this->sender = $sender;
        $this->recipient = $recipient;
        $this->origin = $origin;
        $this->destination = $destination;
        $this->priority = $priority;
        $this->valueAddedServices = $valueAddedServices;
        $this->references = $references;
        $this->parcel = $parcel;
    }

    public function getType(): ShipmentType
    {
        return ShipmentType::PointToAddress();
    }

    public function getSender(): Sender
    {
        return $this->sender;
    }

    public function getRecipient(): Recipient
    {
        return $this->recipient;
    }

    public function getOrigin(): OriginPoint
    {
        return $this->origin;
    }

    public function getDestination(): DestinationAddress
    {
        return $this->destination;
    }

    public function getPriority(): ShipmentPriority
    {
        return $this->priority;
    }

    public function getValueAddedServices(): ?ValueAddedServices
    {
        return $this->valueAddedServices;
    }

    public function getReferences(): ?References
    {
        return $this->references;
    }

    public function getParcel(): Parcel
    {
        return $this->parcel;
    }

    /**
     * @return array<string, mixed>
     */
    public function jsonSerialize(): array
    {
        return get_object_vars($this);
    }
}
