<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Api\Tracking;

use InPost\International\Translation\TranslatableInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class TrackingEvent implements TranslatableInterface
{
    public const CREATED = 'CRE.1001';
    public const READY_FOR_ACCEPTANCE = 'CRE.1002';
    public const READY_FOR_COURIER_COLLECTION = 'FMD.1001';
    public const COLLECTED_BY_COURIER = 'FMD.1002';
    public const IN_TRANSIT_FIRST_MILE = 'FMD.1003';
    public const COLLECTED_BY_COURIER_IN_PUDO = 'FMD.1004';
    public const COLLECTED_BY_COURIER_IN_APM = 'FMD.1005';
    public const PARCEL_CLAIMED = 'FMD.9001';
    public const ADOPTED_AT_LOGISTICS_CENTRE = 'MMD.1001';
    public const PROCESSED_AT_LOGISTICS_CENTRE = 'MMD.1002';
    public const DISPATCHED_FROM_LOGISTICS_CENTRE = 'MMD.1003';
    public const LINE_HAUL = 'MMD.1004';
    public const IN_TRANSIT_LAST_MILE = 'LMD.1001';
    public const ARRIVED_AT_DESTINATION = 'LMD.1002';
    public const READY_TO_COLLECT = 'LMD.1003';
    public const READY_TO_COLLECT_PUDO = 'LMD.1004';
    public const READY_TO_COLLECT_APM = 'LMD.1005';
    public const ALTERNATIVE_TEMPORARY_COLLECTION_POINT_ASSIGNED = 'LMD.3002';
    public const ALTERNATIVE_COLLECTION_POINT_ASSIGNED = 'LMD.3003';
    public const BRANCH_COLLECTION_ASSIGNED = 'LMD.3004';
    public const ORIGINAL_COLLECTION_POINT_REASSIGNED = 'LMD.3005';
    public const DELIVERY_READDRESSED = 'LMD.3006';
    public const STORED_TEMPORARY_IN_SERVICE_POINT = 'LMD.3007';
    public const EXPIRED_STORED_PARCEL = 'LMD.3008';
    public const EXPIRED_ON_TEMPORARY_BOX_MACHINE = 'LMD.3009';
    public const EXPIRED_ON_TEMPORARY_BOX_MACHINE_2 = 'LMD.3010';
    public const EXPIRED_ON_TEMPORARY_COLLECTION_POINT = 'LMD.3011';
    public const REDIRECT_CANCELLED = 'LMD.3012';
    public const REDIRECTED_TO_PUDO = 'LMD.3013';
    public const REDIRECTED_TO_APM = 'LMD.3014';
    public const PARCEL_AT_CUSTOMS = 'CC.001';
    public const PARCEL_CUSTOMS_CLEARED = 'CC.002';
    public const PARCEL_HELD_AT_CUSTOMS = 'CC.003';
    public const REMINDER_TO_COLLECT = 'LMD.9001';
    public const EXPIRED = 'LMD.9002';
    public const OVERSIZED = 'LMD.9003';
    public const ATTEMPTED_DELIVERY = 'LMD.9004';
    public const UNDELIVERABLE = 'LMD.9005';
    public const UNDELIVERABLE_REJECTED_BY_RECIPIENT = 'LMD.9006';
    public const UNDELIVERABLE_INCORRECT_DELIVERY_DETAILS = 'LMD.9007';
    public const UNDELIVERABLE_RECEIVER_UNKNOWN = 'LMD.9008';
    public const UNDELIVERABLE_COD_CONDITIONS_NOT_MET = 'LMD.9009';
    public const UNDELIVERABLE_NO_MAILBOX = 'LMD.9010';
    public const UNDELIVERABLE_NO_ACCESS_TO_LOCATION = 'LMD.9011';
    public const STORED_TEMPORARY_IN_BOX_MACHINE = 'LMD.9012';
    public const PARCEL_READY_TO_COLLECT_AT_CUSTOMER_SERVICE_POINT = 'LMD.9013';
    public const DELIVERED = 'EOL.1001';
    public const PARCEL_COLLECTED = 'EOL.1002';
    public const DELIVERED_AT_SAFE_PLACE = 'EOL.1003';
    public const DELIVERED_AT_NEIGHBOUR = 'EOL.1004';
    public const DELIVERED_WITH_VERIFIED_RECIPIENT = 'EOL.1005';
    public const MISSING = 'EOL.9001';
    public const DAMAGED = 'EOL.9002';
    public const DESTROYED = 'EOL.9003';
    public const CANCELLED = 'EOL.9004';
    public const RETURNING_TO_SENDER = 'RTS.1001';
    public const RETURNED_TO_SENDER = 'RTS.1002';
    public const PICKED = 'FUL.1001';
    public const PACKED = 'FUL.1002';
    public const DISPATCHED = 'FUL.1003';
    public const COD_PAYMENT_RECEIVED = 'INF.1001';
    public const DELAY_IN_DELIVERY = 'INF.9001';
    public const HANDOVER = 'HAN.1001';

    /**
     * @var string
     */
    private $code;

    public function __construct(string $code)
    {
        $this->code = $code;
    }

    /**
     * @return string[]
     */
    public static function getDeliveredEventCodes(): array
    {
        return [
            self::DELIVERED,
            self::PARCEL_COLLECTED,
            self::DELIVERED_AT_SAFE_PLACE,
            self::DELIVERED_AT_NEIGHBOUR,
            self::DELIVERED_WITH_VERIFIED_RECIPIENT,
        ];
    }

    public function getCode(): string
    {
        return $this->code;
    }

    public function trans(TranslatorInterface $translator, string $locale = null): string
    {
        switch ($this->code) {
            case self::CREATED:
                return $translator->trans('Created', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::READY_FOR_ACCEPTANCE:
                return $translator->trans('Ready for acceptance', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::READY_FOR_COURIER_COLLECTION:
                return $translator->trans('Ready for courier collection', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::COLLECTED_BY_COURIER:
                return $translator->trans('Collected by courier', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::IN_TRANSIT_FIRST_MILE:
                return $translator->trans('In-transit (first-mile)', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::COLLECTED_BY_COURIER_IN_PUDO:
                return $translator->trans('Collected by courier in PUDO', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::COLLECTED_BY_COURIER_IN_APM:
                return $translator->trans('Collected by courier in APM', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::PARCEL_CLAIMED:
                return $translator->trans('Parcel claimed', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::ADOPTED_AT_LOGISTICS_CENTRE:
                return $translator->trans('Adopted at Logistics Centre', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::PROCESSED_AT_LOGISTICS_CENTRE:
                return $translator->trans('Processed at Logistics Centre', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::DISPATCHED_FROM_LOGISTICS_CENTRE:
                return $translator->trans('Dispatched from Logistics Centre', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::LINE_HAUL:
                return $translator->trans('Line-Haul', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::IN_TRANSIT_LAST_MILE:
                return $translator->trans('In-transit (last-mile)', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::ARRIVED_AT_DESTINATION:
                return $translator->trans('Arrived at destination', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::READY_TO_COLLECT:
                return $translator->trans('Ready to collect', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::READY_TO_COLLECT_PUDO:
                return $translator->trans('Ready to collect PUDO', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::READY_TO_COLLECT_APM:
                return $translator->trans('Ready to collect APM', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::ALTERNATIVE_TEMPORARY_COLLECTION_POINT_ASSIGNED:
                return $translator->trans('Alternative temporary collection point assigned', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::ALTERNATIVE_COLLECTION_POINT_ASSIGNED:
                return $translator->trans('Alternative collection point assigned', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::BRANCH_COLLECTION_ASSIGNED:
                return $translator->trans('Branch collection assigned', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::ORIGINAL_COLLECTION_POINT_REASSIGNED:
                return $translator->trans('Original collection point reassigned', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::DELIVERY_READDRESSED:
                return $translator->trans('Delivery readdressed', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::STORED_TEMPORARY_IN_SERVICE_POINT:
                return $translator->trans('Stored temporary in a service point', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::EXPIRED_STORED_PARCEL:
                return $translator->trans('Expired stored parcel', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::EXPIRED_ON_TEMPORARY_BOX_MACHINE:
            case self::EXPIRED_ON_TEMPORARY_BOX_MACHINE_2:
                return $translator->trans('Expired on temporary box machine', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::EXPIRED_ON_TEMPORARY_COLLECTION_POINT:
                return $translator->trans('Expired on temporary collection point', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::REDIRECT_CANCELLED:
                return $translator->trans('Redirect cancelled', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::REDIRECTED_TO_PUDO:
                return $translator->trans('Redirected to PUDO', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::REDIRECTED_TO_APM:
                return $translator->trans('Redirected to APM', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::PARCEL_AT_CUSTOMS:
                return $translator->trans('Parcel at customs', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::PARCEL_CUSTOMS_CLEARED:
                return $translator->trans('Parcel customs cleared', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::PARCEL_HELD_AT_CUSTOMS:
                return $translator->trans('Parcel held at customs', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::REMINDER_TO_COLLECT:
                return $translator->trans('Reminder to collect', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::EXPIRED:
                return $translator->trans('Expired', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::OVERSIZED:
                return $translator->trans('Oversized', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::ATTEMPTED_DELIVERY:
                return $translator->trans('Attempted delivery', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::UNDELIVERABLE:
                return $translator->trans('Undeliverable', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::UNDELIVERABLE_REJECTED_BY_RECIPIENT:
                return $translator->trans('Undeliverable: rejected by recipient', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::UNDELIVERABLE_INCORRECT_DELIVERY_DETAILS:
                return $translator->trans('Undeliverable: incorrect delivery details', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::UNDELIVERABLE_RECEIVER_UNKNOWN:
                return $translator->trans('Undeliverable: receiver unknown', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::UNDELIVERABLE_COD_CONDITIONS_NOT_MET:
                return $translator->trans('Undeliverable: COD conditions not met', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::UNDELIVERABLE_NO_MAILBOX:
                return $translator->trans('Undeliverable: no mailbox', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::UNDELIVERABLE_NO_ACCESS_TO_LOCATION:
                return $translator->trans('Undeliverable: no access to location', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::STORED_TEMPORARY_IN_BOX_MACHINE:
                return $translator->trans('Stored temporary in a box machine', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::PARCEL_READY_TO_COLLECT_AT_CUSTOMER_SERVICE_POINT:
                return $translator->trans('Parcel ready to collect at customer service point', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::DELIVERED:
                return $translator->trans('Delivered', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::PARCEL_COLLECTED:
                return $translator->trans('Parcel collected', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::DELIVERED_AT_SAFE_PLACE:
                return $translator->trans('Delivered at Safe Place', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::DELIVERED_AT_NEIGHBOUR:
                return $translator->trans('Delivered at neighbour', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::DELIVERED_WITH_VERIFIED_RECIPIENT:
                return $translator->trans('Delivered with verified recipient', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::MISSING:
                return $translator->trans('Missing', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::DAMAGED:
                return $translator->trans('Damaged', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::DESTROYED:
                return $translator->trans('Destroyed', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::CANCELLED:
                return $translator->trans('Cancelled', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::RETURNING_TO_SENDER:
                return $translator->trans('Returning to sender', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::RETURNED_TO_SENDER:
                return $translator->trans('Returned to sender', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::PICKED:
                return $translator->trans('Fulfillment: picked', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::PACKED:
                return $translator->trans('Fulfillment: packed', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::DISPATCHED:
                return $translator->trans('Fulfillment: dispatched', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::COD_PAYMENT_RECEIVED:
                return $translator->trans('COD payment received', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::DELAY_IN_DELIVERY:
                return $translator->trans('Delay in delivery', [], 'Modules.Inpostinternational.Tracking', $locale);
            case self::HANDOVER:
                return $translator->trans('Handed over to third party', [], 'Modules.Inpostinternational.Tracking', $locale);
            default:
                return $this->code;
        }
    }
}
