<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Carrier;

use InPost\International\Api\Shipment\ShipmentType;
use InPost\International\Country;
use InPost\International\Entity\CourierCarrier;
use InPost\International\Entity\PointDeliveryCarrier;
use InPost\International\Enum\StringEnum;
use InPost\International\Translation\TranslatableInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * @method static self Courier()
 * @method static self PointDelivery()
 */
final class CarrierType extends StringEnum implements TranslatableInterface
{
    private const COURIER = CourierCarrier::TYPE;
    private const POINT_DELIVERY = PointDeliveryCarrier::TYPE;

    public static function fromShipmentType(ShipmentType $shipmentType): self
    {
        switch ($shipmentType) {
            case ShipmentType::PointToAddress():
            case ShipmentType::AddressToAddress():
                return self::Courier();
            case ShipmentType::PointToPoint():
            case ShipmentType::AddressToPoint():
                return self::PointDelivery();
            default:
                throw new \LogicException('Not implemented.');
        }
    }

    public function trans(TranslatorInterface $translator, string $locale = null): string
    {
        switch ($this) {
            case self::Courier():
                return $translator->trans('Courier', [], 'Modules.Inpostinternational.Carrier', $locale);
            case self::PointDelivery():
                return $translator->trans('Point delivery', [], 'Modules.Inpostinternational.Carrier', $locale);
            default:
                return $this->name;
        }
    }

    /**
     * @return ShipmentType[]
     */
    public function getPossibleShipmentTypes(): array
    {
        switch ($this) {
            case self::Courier():
                return [ShipmentType::PointToAddress(), ShipmentType::AddressToAddress()];
            case self::PointDelivery():
                return [ShipmentType::PointToPoint(), ShipmentType::AddressToPoint()];
            default:
                throw new \LogicException('Not implemented.');
        }
    }

    /**
     * @return Country[]
     */
    public function getPossibleDestinationCountries(): array
    {
        switch ($this) {
            case self::Courier():
                return [
                    Country::Hungary(),
                    Country::Austria(),
                ];
            case self::PointDelivery():
                return array_filter(Country::cases(), static function (Country $country) {
                    return !in_array($country, [
                        Country::Poland(),
                        // TODO: Hungary should be available in the near future
                        Country::Hungary(),
                    ], true);
                });
            default:
                throw new \LogicException('Not implemented.');
        }
    }

    /**
     * @return float weight in kilograms
     */
    public function getMaxParcelWeight(Country $country): float
    {
        switch ($this) {
            case self::Courier():
                return 30.;
            case self::PointDelivery():
                return self::getMaxAPMParcelWeight($country);
            default:
                throw new \LogicException('Not implemented.');
        }
    }

    private static function getMaxAPMParcelWeight(Country $country): float
    {
        if (Country::Hungary() === $country) {
            return 12.;
        }

        return 25.;
    }
}
