<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Carrier\Configurator;

use Doctrine\ORM\EntityRepository;
use InPost\International\Carrier\CarrierConfiguration;
use InPost\International\Carrier\CarrierType;
use InPost\International\Carrier\Message\CreateCarrierCommand;
use InPost\International\Country;
use InPost\International\Shipment\TrackingUrlGenerator;
use PrestaShopBundle\Entity\Lang;
use Symfony\Contracts\Translation\TranslatorInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class BasicConfigurator implements CarrierConfiguratorInterface
{
    /**
     * @var TranslatorInterface
     */
    private $translator;

    /**
     * @var \CarrierModule
     */
    private $module;

    /**
     * @var EntityRepository<Lang>
     */
    private $languageRepository;

    /**
     * @var array<int, string>|null
     */
    private $localeByLanguageId;

    /**
     * @param EntityRepository<Lang> $languageRepository
     */
    public function __construct(TranslatorInterface $translator, \CarrierModule $module, EntityRepository $languageRepository)
    {
        $this->translator = $translator;
        $this->module = $module;
        $this->languageRepository = $languageRepository;
    }

    public function configure(CarrierConfiguration $configuration, CreateCarrierCommand $command): void
    {
        $type = $command->getType();
        $carrier = $configuration->getCarrier();

        $domestic = [Country::Poland()] === $countries = $command->getCountries();

        $carrier->is_module = true;
        $carrier->external_module_name = $this->module->name;
        $carrier->name = $command->getName();
        $carrier->url = TrackingUrlGenerator::generate('@', $domestic ? 'pl' : 'en');
        $carrier->delay = $this->getDelay($type);
        $carrier->shipping_external = true;
        $carrier->need_range = true;
        $carrier->shipping_method = \Carrier::SHIPPING_METHOD_WEIGHT;
        $carrier->max_weight = self::getMaxWeight($type, $countries);
        $carrier->range_behavior = true;

        $configuration->setLogoPath($this->module->getLocalPath() . 'views/img/carrier_logo.png');
    }

    private function getDelay(CarrierType $type): array
    {
        return array_map(function (string $locale) use ($type): string {
            return $this->translator->trans('{carrier_type}: 3 - 5 working days', [
                '{carrier_type}' => $type->trans($this->translator, $locale),
            ], 'Modules.Inpostinternational.Carrier', $locale);
        }, $this->getLocaleByLanguageId());
    }

    private function getLocaleByLanguageId(): array
    {
        if (isset($this->localeByLanguageId)) {
            return $this->localeByLanguageId;
        }

        $localeByLanguageId = [];

        foreach ($this->languageRepository->findAll() as $language) {
            $localeByLanguageId[$language->getId()] = $language->getLocale();
        }

        return $this->localeByLanguageId = $localeByLanguageId;
    }

    /**
     * @param Country[] $countries
     */
    private static function getMaxWeight(CarrierType $type, array $countries): float
    {
        $maxWeights = array_map([$type, 'getMaxParcelWeight'], $countries);

        return min($maxWeights);
    }
}
