<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Carrier\Form;

use InPost\International\Carrier\CarrierType;
use InPost\International\Carrier\Message\CreateCarrierCommand;
use InPost\International\Country;
use InPost\International\Enum\Form\Type\EnumType;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\Form\FormView;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Contracts\Translation\TranslatorInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class CreateCarrierType extends AbstractType
{
    /**
     * @var TranslatorInterface
     */
    private $translator;

    public function __construct(TranslatorInterface $translator)
    {
        $this->translator = $translator;
    }

    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('type', EnumType::class, [
                'class' => CarrierType::class,
                'label' => $this->translator->trans('Type', [], 'Modules.Inpostinternational.Carrier'),
                'attr' => [
                    'class' => 'js-inpost-intl-carrier-type-choice',
                    'autocomplete' => 'off',
                ],
            ])
            ->add('name', TextType::class, [
                'label' => $this->translator->trans('Name', [], 'Admin.Global'),
            ])
            ->addEventListener(FormEvents::PRE_SET_DATA, function (FormEvent $event) {
                $data = $event->getData();
                $carrierType = $data instanceof CreateCarrierCommand ? $data->getType() : null;

                $this->addCountriesChoice(
                    $event->getForm(),
                    $carrierType ?? CarrierType::cases()[0]
                );
            });

        $builder->get('type')->addEventListener(FormEvents::POST_SUBMIT, function (FormEvent $event) {
            $form = $event->getForm();
            $data = $form->getData();

            if (!$data instanceof CarrierType) {
                return;
            }

            $this->addCountriesChoice($form->getParent(), $data);
        });
    }

    public function finishView(FormView $view, FormInterface $form, array $options): void
    {
        $view['countries']->vars['isCheckSelectAll'] = false;
        $view['countries']->vars['displayTotalItems'] = false;
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => CreateCarrierCommand::class,
        ]);
    }

    private function addCountriesChoice(FormInterface $form, CarrierType $type): void
    {
        $form->add('countries', EnumType::class, [
            'class' => Country::class,
            'choices' => $type->getPossibleDestinationCountries(),
            'multiple' => true,
            'expanded' => true,
            'label' => $this->translator->trans('Countries', [], 'Admin.Global'),
            'block_prefix' => 'material_choice_table',
        ]);
    }
}
