<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Checkout\Form;

use InPost\International\Checkout\CheckoutSessionRepositoryInterface;
use InPost\International\Checkout\Form\Model\PointDeliveryOptions;
use InPost\International\Checkout\Form\Type\PointDeliveryOptionsType;
use InPost\International\Checkout\Message\UpdateCheckoutSessionCommand;
use InPost\International\Delivery\Util\CarrierFinder;
use InPost\International\Delivery\Util\DeliveryCountryResolver;
use InPost\International\Entity\Carrier;
use InPost\International\Entity\CheckoutSession;
use InPost\International\Entity\PointDeliveryCarrier;
use Symfony\Component\Form\Extension\Core\Type\FormType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\Form\FormInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class DeliveryOptionsFormFactory implements DeliveryOptionsFormFactoryInterface
{
    /**
     * @var FormFactoryInterface
     */
    private $formFactory;

    /**
     * @var CarrierFinder
     */
    private $carrierFinder;

    /**
     * @var DeliveryCountryResolver
     */
    private $countryResolver;

    /**
     * @var CheckoutSessionRepositoryInterface
     */
    private $sessionRepository;

    public function __construct(FormFactoryInterface $formFactory, CarrierFinder $carrierFinder, DeliveryCountryResolver $countryResolver, CheckoutSessionRepositoryInterface $sessionRepository)
    {
        $this->formFactory = $formFactory;
        $this->carrierFinder = $carrierFinder;
        $this->countryResolver = $countryResolver;
        $this->sessionRepository = $sessionRepository;
    }

    public function getForCarrier(\Cart $cart, Carrier $carrier): ?FormInterface
    {
        if (!$carrier instanceof PointDeliveryCarrier) {
            return null;
        }

        return $this->buildForm($cart, $carrier)->getForm();
    }

    /**
     * {@inheritDoc}
     */
    public function getForDeliveryOption(\Cart $cart, array $deliveryOption): ?FormInterface
    {
        if ([] === $carriers = $this->carrierFinder->findByDeliveryOption($deliveryOption)) {
            return null;
        }

        return $this->buildForm($cart, ...$carriers)->getForm();
    }

    private function buildForm(\Cart $cart, Carrier ...$carriers): FormBuilderInterface
    {
        if (null === $country = $this->countryResolver->resolve($cart)) {
            throw new \LogicException('Unable to resolve delivery country.');
        }

        $command = $this->createUpdateCommand((int) $cart->id, ...$carriers);

        $builder = $this->formFactory->createNamedBuilder('inpost_intl_options', FormType::class, $command, [
            'allow_extra_fields' => true,
        ]);

        foreach ($carriers as $carrier) {
            if (!$carrier instanceof PointDeliveryCarrier) {
                continue;
            }

            $builder->add((string) $carrier->getReferenceId(), PointDeliveryOptionsType::class, [
                'country' => $country,
                'property_path' => sprintf('deliveryOptions[%d]', $carrier->getReferenceId()),
            ]);
        }

        return $builder;
    }

    private function createUpdateCommand(int $cartId, Carrier ...$carriers): UpdateCheckoutSessionCommand
    {
        $command = new UpdateCheckoutSessionCommand($cartId);
        $session = $this->sessionRepository->find($cartId);

        foreach ($carriers as $carrier) {
            $this->addDeliveryOptions($command, $carrier, $session);
        }

        return $command;
    }

    private function addDeliveryOptions(UpdateCheckoutSessionCommand $command, Carrier $carrier, ?CheckoutSession $session): void
    {
        if (!$carrier instanceof PointDeliveryCarrier) {
            return;
        }

        $options = new PointDeliveryOptions();

        if (null !== $session) {
            $options->setPointId($session->getPointId($carrier->getReferenceId()));
        }

        $command->setPointDeliveryOptions($carrier->getReferenceId(), $options);
    }
}
