<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Checkout\Module;

use InPost\International\Asset\Assets;
use InPost\International\Asset\Provider\AssetsProviderInterface;
use InPost\International\Asset\Provider\GeoWidgetAssetsProvider;
use InPost\International\Checkout\ModuleCheckoutInterface;
use InPost\International\Checkout\View\CheckoutViewRendererInterface;
use InPost\International\PrestaShop\Module\Util\ControllerChecker;
use Symfony\Component\HttpFoundation\Request;

if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * The module does not offer any real way to validate/process carrier data before finalizing the order.
 */
final class WebKulOnePageCheckout implements ModuleCheckoutInterface
{
    private const MODULE_NAME = 'wkonepagecheckout';

    /**
     * @var AssetsProviderInterface
     */
    private $assetsProvider;

    /**
     * @var CheckoutViewRendererInterface
     */
    private $formRenderer;

    public function __construct(AssetsProviderInterface $assetsProvider, CheckoutViewRendererInterface $formRenderer)
    {
        $this->assetsProvider = $assetsProvider;
        $this->formRenderer = $formRenderer;
    }

    public static function getModuleName(): string
    {
        return self::MODULE_NAME;
    }

    public static function supports(\FrontControllerCore $controller, Request $request): bool
    {
        if ($controller instanceof \OrderControllerCore) {
            return true;
        }

        return ControllerChecker::isModuleController($controller, self::MODULE_NAME, [
            'wkcheckout',
        ]);
    }

    public static function getSubscribedEvents(): array
    {
        return [];
    }

    public function getAssets(): Assets
    {
        $assets = $this->assetsProvider->getAssets();

        if (null === $assets->getJavaScriptVariable(GeoWidgetAssetsProvider::GEO_WIDGET_CONFIG_VAR_NAME)) {
            return $assets;
        }

        return $assets->addJavaScript('js/front/wk-one-page-checkout.js');
    }

    public function renderView(Request $request, \Cart $cart, int $carrierId): string
    {
        return $this->formRenderer->render($cart, $carrierId);
    }
}
