<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Configuration\Form\Type\Api;

use InPost\International\OAuth2\Authentication\ClientCredentials;
use InPost\International\OAuth2\Authentication\ClientCredentialsInterface;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\DataMapperInterface;
use Symfony\Component\Form\Exception\UnexpectedTypeException;
use Symfony\Component\Form\Extension\Core\Type\PasswordType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\Form\FormView;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Contracts\Translation\TranslatorInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class ClientCredentialsType extends AbstractType implements DataMapperInterface
{
    private const MASKED_VALUE = '*****';

    /**
     * @var TranslatorInterface
     */
    private $translator;

    public function __construct(TranslatorInterface $translator)
    {
        $this->translator = $translator;
    }

    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('clientId', TextType::class, [
                'label' => $this->translator->trans('Client ID', [], 'Modules.Inpostinternational.Configuration'),
            ])
            ->add('clientSecret', PasswordType::class, [
                'label' => $this->translator->trans('Client secret', [], 'Modules.Inpostinternational.Configuration'),
                'always_empty' => false,
            ])
            ->setDataMapper($this);
    }

    public function finishView(FormView $view, FormInterface $form, array $options): void
    {
        if (!$form->get('clientSecret')->getData()) {
            return;
        }

        $childView = $view['clientSecret'];

        if ('' === $childView->vars['value']) {
            $childView->vars['value'] = self::MASKED_VALUE;
        }
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => ClientCredentials::class,
            'empty_data' => null,
        ]);
    }

    /**
     * @param ClientCredentialsInterface|null $viewData
     * @param iterable<FormInterface> $forms
     */
    public function mapDataToForms($viewData, $forms): void
    {
        if (null === $viewData) {
            return;
        }

        if (!$viewData instanceof ClientCredentialsInterface) {
            throw new UnexpectedTypeException($viewData, ClientCredentialsInterface::class);
        }

        if (!is_array($forms)) {
            $forms = iterator_to_array($forms);
        }

        $forms['clientId']->setData($viewData->getClientId());
        $forms['clientSecret']->setData($viewData->getClientSecret());
    }

    /**
     * @param iterable<FormInterface> $forms
     * @param ClientCredentialsInterface|null $viewData
     */
    public function mapFormsToData($forms, &$viewData): void
    {
        if (!is_array($forms)) {
            $forms = iterator_to_array($forms);
        }

        $clientId = $forms['clientId']->getData();
        $clientSecret = $forms['clientSecret']->getData();

        if ($this->areSameCredentials($viewData, $clientId, $clientSecret)) {
            return;
        }

        if (null === $clientId && null === $clientSecret) {
            $viewData = null;
        } else {
            $viewData = new ClientCredentials($clientId ?? '', $clientSecret);
        }
    }

    /**
     * @param mixed $credentials
     */
    private function areSameCredentials($credentials, ?string $clientId, ?string $clientSecret): bool
    {
        if (null === $credentials) {
            return false;
        }

        if (!$credentials instanceof ClientCredentialsInterface) {
            throw new UnexpectedTypeException($credentials, ClientCredentialsInterface::class);
        }

        if ($clientId !== $credentials->getClientId()) {
            return false;
        }

        return $clientSecret === $credentials->getClientSecret() || self::MASKED_VALUE === $clientSecret;
    }
}
