<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Configuration\Form\Type\Api;

use InPost\International\Configuration\EnvironmentConfiguration;
use InPost\International\Environment\EnvironmentInterface;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\DataMapperInterface;
use Symfony\Component\Form\Exception\UnexpectedTypeException;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Contracts\Translation\TranslatorInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class EnvironmentConfigurationType extends AbstractType implements DataMapperInterface
{
    /**
     * @var TranslatorInterface
     */
    private $translator;

    public function __construct(TranslatorInterface $translator)
    {
        $this->translator = $translator;
    }

    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('credentials', ClientCredentialsType::class, [
                'required' => false,
                'label' => false,
            ])
            ->add('geoWidgetToken', TextareaType::class, [
                'required' => false,
                'label' => $this->translator->trans('GeoWidget token', [], 'Modules.Inpostinternational.Configuration'),
            ])
            ->setDataMapper($this);
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver
            ->setDefaults([
                'data_class' => EnvironmentConfiguration::class,
                'empty_data' => null,
                'error_bubbling' => false,
            ])
            ->setRequired('environment')
            ->setAllowedTypes('environment', EnvironmentInterface::class);
    }

    /**
     * @param EnvironmentConfiguration|null $viewData
     * @param iterable<FormInterface> $forms
     */
    public function mapDataToForms($viewData, $forms): void
    {
        if (null === $viewData) {
            return;
        }

        if (!$viewData instanceof EnvironmentConfiguration) {
            throw new UnexpectedTypeException($viewData, EnvironmentConfiguration::class);
        }

        $forms = is_array($forms) ? $forms : iterator_to_array($forms);

        $forms['credentials']->setData($viewData->getClientCredentials());
        $forms['geoWidgetToken']->setData($viewData->getGeoWidgetToken());
    }

    /**
     * @param iterable<FormInterface> $forms
     * @param EnvironmentConfiguration|null $viewData
     */
    public function mapFormsToData($forms, &$viewData): void
    {
        $forms = is_array($forms) ? $forms : iterator_to_array($forms);
        $environment = current($forms)->getParent()->getConfig()->getOption('environment');

        $viewData = new EnvironmentConfiguration(
            $environment,
            $forms['credentials']->getData(),
            $forms['geoWidgetToken']->getData()
        );
    }
}
