<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Configuration\Form\Type;

use InPost\International\Configuration\Message\UpdateOrdersConfigurationCommand;
use InPost\International\PrestaShop\ObjectModel\Form\Type\ObjectModelType;
use InPost\International\Shipment\Command\UpdateShipmentsCommand;
use PrestaShopBundle\Form\Admin\Type\SwitchType;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Contracts\Translation\TranslatorInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class OrdersConfigurationType extends AbstractType
{
    /**
     * @var TranslatorInterface
     */
    private $translator;

    public function __construct(TranslatorInterface $translator)
    {
        $this->translator = $translator;
    }

    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('labelPrintedStatusId', ObjectModelType::class, [
                'class' => \OrderState::class,
                'input' => 'id',
                'required' => false,
                'label' => $this->translator->trans('Order status: label printed', [], 'Modules.Inpostinternational.Configuration'),
                'placeholder' => $this->translator->trans('Do not change order status', [], 'Modules.Inpostinternational.Configuration'),
            ])
            ->add('shipmentDeliveredStatusId', ObjectModelType::class, [
                'class' => \OrderState::class,
                'input' => 'id',
                'required' => false,
                'label' => $this->translator->trans('Order status: shipment delivered', [], 'Modules.Inpostinternational.Configuration'),
                'placeholder' => $this->translator->trans('Do not change order status', [], 'Modules.Inpostinternational.Configuration'),
                'help' => nl2br(implode("\n", [
                    htmlspecialchars($this->translator->trans('You can use the "{command}" console command to set up a CRON job that will periodically fetch shipment status data from the API. The "--environment" option allows to specify the API environment to use ("prod" or "sandbox", defaults to "prod"). E.g.:', [
                        '{command}' => UpdateShipmentsCommand::NAME,
                    ], 'Modules.Inpostinternational.Configuration')),
                    sprintf('<code>php bin/console %s --environment=sandbox</code>', UpdateShipmentsCommand::NAME),
                ])),
                'help_html' => true,
            ])
            ->add('modifyOrderConfirmationMail', SwitchType::class, [
                'label' => $this->translator->trans('Modify order confirmation email', [], 'Modules.Inpostinternational.Configuration'),
                'label_attr' => [
                    'popover' => $this->translator->trans('If enabled, delivery point name will be appended to the carrier name.', [], 'Modules.Inpostinternational.Configuration'),
                ],
            ]);
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => UpdateOrdersConfigurationCommand::class,
        ]);
    }
}
