<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Configuration\Form\Type;

use InPost\International\Api\Shipment\Model\Service\Currency;
use InPost\International\Common\Form\Type\ContactDetailsType;
use InPost\International\Configuration\Enum\ShipmentReference;
use InPost\International\Configuration\Message\UpdateShippingConfigurationCommand;
use InPost\International\Enum\Form\Type\EnumType;
use InPost\International\Shipment\ShippingMethod;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\MoneyType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Contracts\Translation\TranslatorInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class ShippingConfigurationType extends AbstractType
{
    /**
     * @var TranslatorInterface
     */
    private $translator;

    public function __construct(TranslatorInterface $translator)
    {
        $this->translator = $translator;
    }

    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('sender', ContactDetailsType::class, [
                'label' => $this->translator->trans('Sender details', [], 'Modules.Inpostinternational.Configuration'),
            ])
            ->add('defaultShippingMethod', EnumType::class, [
                'class' => ShippingMethod::class,
                'label' => $this->translator->trans('Default shipping method', [], 'Modules.Inpostinternational.Configuration'),
            ])
            ->add('defaultShipmentReference', EnumType::class, [
                'class' => ShipmentReference::class,
                'label' => $this->translator->trans('Default reference', [], 'Modules.Inpostinternational.Configuration'),
                'label_attr' => [
                    'popover' => $this->translator->trans('The selected value will be added to the API data as a custom shipment reference with name "order". Additional references may be added via the shipment creation form.', [], 'Modules.Inpostinternational.Configuration'),
                ],
            ])
            ->add('defaultParcelComment', EnumType::class, [
                'class' => ShipmentReference::class,
                'required' => false,
                'placeholder' => $this->translator->trans('None', [], 'Modules.Inpostinternational.Admin'),
                'label' => $this->translator->trans('Default customer reference', [], 'Modules.Inpostinternational.Configuration'),
                'label_attr' => [
                    'popover' => $this->translator->trans('The content will be printed on the parcel label.', [], 'Modules.Inpostinternational.Configuration'),
                ],
            ])
            ->add('defaultInsurance', MoneyType::class, [
                'required' => false,
                'scale' => 2,
                'currency' => Currency::getDefault()->value,
                'label' => $this->translator->trans('Default insurance', [], 'Modules.Inpostinternational.Configuration'),
            ]);
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => UpdateShippingConfigurationCommand::class,
        ]);
    }
}
