<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Configuration\Message;

use InPost\International\Common\DTO\ContactDetails;
use InPost\International\Common\DTO\Phone;
use InPost\International\Configuration\Enum\ShipmentReference;
use InPost\International\Configuration\ShippingConfiguration;
use InPost\International\Shipment\ShippingMethod;
use Symfony\Component\Validator\Constraints as Assert;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class UpdateShippingConfigurationCommand
{
    /**
     * @var ContactDetails|null
     *
     * @Assert\NotNull
     * @Assert\Valid
     */
    private $sender;

    /**
     * @var ShippingMethod|null
     *
     * @Assert\NotBlank
     */
    private $defaultShippingMethod;

    /**
     * @var ShipmentReference|null
     *
     * @Assert\NotBlank
     */
    private $defaultShipmentReference;

    /**
     * @var ShipmentReference|null
     */
    private $defaultParcelComment;

    /**
     * @var float|null
     *
     * @Assert\Range(min=0, max=5000)
     */
    private $defaultInsurance;

    public static function forConfiguration(ShippingConfiguration $configuration): self
    {
        $command = new self();

        $sender = $configuration->getSender();
        $phone = $sender->getPhone();

        $phoneDto = (new Phone())
            ->setPrefix($phone->getPrefix())
            ->setNumber($phone->getNumber());

        $senderDto = (new ContactDetails())
            ->setCompanyName($sender->getCompanyName())
            ->setFirstName($sender->getFirstName())
            ->setLastName($sender->getLastName())
            ->setEmail($sender->getEmail())
            ->setPhone($phoneDto);

        $command->setSender($senderDto);
        $command->setDefaultShippingMethod($configuration->getDefaultShippingMethod());
        $command->setDefaultShipmentReference($configuration->getDefaultShipmentReference());
        $command->setDefaultParcelComment($configuration->getDefaultParcelComment());

        if (null !== $insurance = $configuration->getDefaultInsurance()) {
            $command->setDefaultInsurance($insurance->getValue());
        }

        return $command;
    }

    public function getSender(): ?ContactDetails
    {
        return $this->sender;
    }

    public function setSender(?ContactDetails $sender): void
    {
        $this->sender = $sender;
    }

    public function getDefaultShippingMethod(): ?ShippingMethod
    {
        return $this->defaultShippingMethod;
    }

    public function setDefaultShippingMethod(?ShippingMethod $defaultShippingMethod): void
    {
        $this->defaultShippingMethod = $defaultShippingMethod;
    }

    public function getDefaultShipmentReference(): ?ShipmentReference
    {
        return $this->defaultShipmentReference;
    }

    public function setDefaultShipmentReference(?ShipmentReference $defaultShipmentReference): void
    {
        $this->defaultShipmentReference = $defaultShipmentReference;
    }

    public function getDefaultParcelComment(): ?ShipmentReference
    {
        return $this->defaultParcelComment;
    }

    public function setDefaultParcelComment(?ShipmentReference $defaultParcelComment): void
    {
        $this->defaultParcelComment = $defaultParcelComment;
    }

    public function getDefaultInsurance(): ?float
    {
        return $this->defaultInsurance;
    }

    public function setDefaultInsurance(?float $defaultInsurance): void
    {
        $this->defaultInsurance = $defaultInsurance;
    }
}
