<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Configuration\Validator;

use InPost\International\Api\Pickup\PickupsApiClientInterface;
use InPost\International\Api\Point\PointsApiClientInterface;
use InPost\International\Api\Shipment\ShipmentsApiClientInterface;
use InPost\International\Api\Tracking\TrackingApiClientInterface;
use InPost\International\Configuration\EnvironmentConfiguration;
use InPost\International\OAuth2\AuthorizationProviderFactoryInterface;
use InPost\International\OAuth2\Exception\AccessTokenRequestException;
use InPost\International\OAuth2\Token\AccessTokenInterface;
use Psr\Http\Client\NetworkExceptionInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class ApiCredentialsValidator extends ConstraintValidator
{
    private const REQUIRED_SCOPES = [
        PointsApiClientInterface::SCOPE_POINTS_READ,
        ShipmentsApiClientInterface::SCOPE_SHIPMENTS_WRITE,
        TrackingApiClientInterface::SCOPE_TRACKING_READ,
        PickupsApiClientInterface::SCOPE_PICKUPS_WRITE,
    ];

    /**
     * @var AuthorizationProviderFactoryInterface
     */
    private $authProviderFactory;

    public function __construct(AuthorizationProviderFactoryInterface $authProviderFactory)
    {
        $this->authProviderFactory = $authProviderFactory;
    }

    public function validate($value, Constraint $constraint): void
    {
        if (!$constraint instanceof ApiCredentials) {
            throw new UnexpectedTypeException($constraint, ApiCredentials::class);
        }

        if (null === $value) {
            return;
        }

        if (!$value instanceof EnvironmentConfiguration) {
            throw new UnexpectedTypeException($value, EnvironmentConfiguration::class);
        }

        if (null === $credentials = $value->getClientCredentials()) {
            return;
        }

        $uriCollection = $value->getEnvironment()->getOAuth2UriCollection();

        try {
            $token = $this->authProviderFactory
                ->create($uriCollection, $credentials)
                ->getAccessToken();

            $this->validateTokenScopes($token);
        } catch (NetworkExceptionInterface $e) {
            $this->context
                ->buildViolation('Could not connect to the authorization server.')
                ->setTranslationDomain('Modules.Inpostinternational.Validators')
                ->addViolation();
        } catch (AccessTokenRequestException $e) {
            $this->addViolationForTokenError($e);
        }
    }

    private function validateTokenScopes(AccessTokenInterface $token): void
    {
        if (null === $scopes = $token->getScopes()) {
            return;
        }

        if ([] === array_diff(self::REQUIRED_SCOPES, $scopes)) {
            return;
        }

        $this->context
            ->buildViolation('The granted access token does not have all of the required scopes.')
            ->atPath('clientCredentials.clientId')
            ->setTranslationDomain('Modules.Inpostinternational.Validators')
            ->addViolation();
    }

    private function addViolationForTokenError(AccessTokenRequestException $e): void
    {
        if (null === $description = $e->getErrorDescription()) {
            $this->context
                ->buildViolation('Could not obtain an access token (error code: "{{ code }}").')
                ->setParameter('{{ code }}', $e->getError())
                ->setTranslationDomain('Modules.Inpostinternational.Validators')
                ->atPath('clientCredentials.clientSecret')
                ->addViolation();

            return;
        }

        $this->context
            ->buildViolation('Could not obtain an access token: {{ description }} (code: "{{ code }}").')
            ->setParameter('{{ code }}', $e->getError())
            ->setParameter('{{ description }}', $description)
            ->setTranslationDomain('Modules.Inpostinternational.Validators')
            ->atPath('clientCredentials.clientSecret')
            ->addViolation();
    }
}
