<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Delivery\Point\Cache;

use InPost\International\Api\Point\Model\Address;
use InPost\International\Api\Point\Model\PointCapability;
use InPost\International\Delivery\Point\RelativePointInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class SimpleRelativePoint implements RelativePointInterface
{
    /**
     * @var string
     */
    private $id;

    /**
     * @var string
     */
    private $type;

    /**
     * @var string
     */
    private $country;

    /**
     * @var string[]
     */
    private $capabilities;

    /**
     * @var Address
     */
    private $address;

    /**
     * @var int
     */
    private $distance;

    /**
     * @param string[] $capabilities
     */
    public function __construct(string $id, string $type, string $country, array $capabilities, Address $address, int $distance)
    {
        $this->id = $id;
        $this->type = $type;
        $this->country = $country;
        $this->capabilities = $capabilities;
        $this->address = $address;
        $this->distance = $distance;
    }

    public static function fromPoint(RelativePointInterface $point): self
    {
        return new self(
            $point->getId(),
            $point->getType(),
            $point->getCountry(),
            $point->getCapabilities(),
            $point->getAddress(),
            $point->getDistance()
        );
    }

    public function getId(): string
    {
        return $this->id;
    }

    public function getType(): string
    {
        return $this->type;
    }

    public function getCountry(): string
    {
        return $this->country;
    }

    public function getCapabilities(): array
    {
        return $this->capabilities;
    }

    public function hasCapability(PointCapability $capability): bool
    {
        return in_array($capability->value, $this->capabilities, true);
    }

    public function getAddress(): Address
    {
        return $this->address;
    }

    public function getDistance(): int
    {
        return $this->distance;
    }
}
