<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Entity;

use Doctrine\ORM\Mapping as ORM;
use InPost\International\Api\Shipment\Model\Parcel\DimensionUnit;
use InPost\International\Shipment\Exception\ParcelException;

if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * @ORM\Embeddable
 */
final class Dimensions implements \Stringable
{
    /**
     * @var float
     *
     * @ORM\Column(type="float")
     */
    private $length;

    /**
     * @var float
     *
     * @ORM\Column(type="float")
     */
    private $width;

    /**
     * @var float
     *
     * @ORM\Column(type="float")
     */
    private $height;

    /**
     * @var string
     *
     * @ORM\Column(type="string")
     */
    private $lengthUnit;

    public function __construct(float $length, float $width, float $height, DimensionUnit $unit)
    {
        if (0. > $length || 0. > $width || 0. > $height) {
            throw new ParcelException('Dimensions cannot be negative.');
        }

        $this->length = $length;
        $this->width = $width;
        $this->height = $height;
        $this->lengthUnit = $unit->value;
    }

    public function getLength(): float
    {
        return $this->length;
    }

    public function getWidth(): float
    {
        return $this->width;
    }

    public function getHeight(): float
    {
        return $this->height;
    }

    public function getUnit(): DimensionUnit
    {
        return DimensionUnit::from($this->lengthUnit);
    }

    public function __toString(): string
    {
        return sprintf('%sｘ%sｘ%s %s', $this->length, $this->width, $this->height, $this->getUnit()->getSymbol());
    }
}
