<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Entity;

use Doctrine\ORM\Mapping as ORM;
use InPost\International\Api\Shipment\Model\Parcel\ParcelType;
use InPost\International\ParcelTemplate\ParcelTemplateRepository;

if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * @final
 *
 * @ORM\Entity(repositoryClass=ParcelTemplateRepository::class)
 * @ORM\Table(name=ParcelTemplate::TABLE_NAME)
 */
class ParcelTemplate implements \Stringable
{
    /**
     * @internal
     */
    public const TABLE_NAME = _DB_PREFIX_ . 'inpost_intl_parcel_template';

    /**
     * @var int|null
     *
     * @ORM\Id
     * @ORM\GeneratedValue
     * @ORM\Column(type="integer")
     */
    private $id;

    /**
     * @var string
     *
     * @ORM\Column(type="string")
     */
    private $name;

    /**
     * @var string
     *
     * @ORM\Column(type="string")
     */
    private $type;

    /**
     * @var Dimensions
     *
     * @ORM\Embedded(class=Dimensions::class, columnPrefix=false)
     */
    private $dimensions;

    /**
     * @var Weight
     *
     * @ORM\Embedded(class=Weight::class, columnPrefix=false)
     */
    private $weight;

    /**
     * @var bool
     *
     * @ORM\Column(type="boolean", name="is_default")
     */
    private $default = false;

    public function __construct(string $name, ParcelType $type, Dimensions $dimensions, Weight $weight)
    {
        $this->setName($name);
        $this->setType($type);
        $this->dimensions = $dimensions;
        $this->weight = $weight;
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getName(): string
    {
        return $this->name;
    }

    public function setName(string $name): void
    {
        if ('' === $name = trim($name)) {
            throw new \InvalidArgumentException('Template name cannot be empty.');
        }

        $this->name = $name;
    }

    public function getType(): ParcelType
    {
        return ParcelType::from($this->type);
    }

    public function setType(ParcelType $type): void
    {
        $this->type = $type->value;
    }

    public function getDimensions(): Dimensions
    {
        return $this->dimensions;
    }

    public function setDimensions(Dimensions $dimensions): void
    {
        $this->dimensions = $dimensions;
    }

    public function getWeight(): Weight
    {
        return $this->weight;
    }

    public function setWeight(Weight $weight): void
    {
        $this->weight = $weight;
    }

    public function isDefault(): bool
    {
        return $this->default;
    }

    public function setDefault(bool $default): void
    {
        $this->default = $default;
    }

    public function __toString(): string
    {
        return sprintf('%s (%s, %s)', $this->name, $this->dimensions, $this->weight);
    }
}
