<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Environment;

use InPost\International\OAuth2\UriCollection;
use InPost\International\OAuth2\UriCollectionInterface;
use Symfony\Component\Translation\IdentityTranslator;
use Symfony\Contracts\Translation\TranslatorInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class ProductionEnvironment implements EnvironmentInterface
{
    public const ID = 'prod';

    private const API_URI = 'https://api.inpost-group.com';
    private const OAUTH2_AUTH_ENDPOINT_URI = 'https://login.inpost-group.com/realms/external/protocol/openid-connect/auth';
    private const OAUTH2_TOKEN_ENDPOINT_URI = self::API_URI . '/auth/token';
    private const GEO_WIDGET_URI = 'https://geowidget.inpost-group.com';

    /**
     * @var TranslatorInterface|null
     */
    private $translator;

    /**
     * @var UriCollection|null
     */
    private $oAuth2Uris;

    public function __construct(TranslatorInterface $translator = null)
    {
        $this->translator = $translator ?? new IdentityTranslator();
    }

    public function getId(): string
    {
        return self::ID;
    }

    public function getName(): string
    {
        return $this->translator->trans('Production', [], 'Modules.Inpostinternational.Environment');
    }

    public function isTestEnvironment(): bool
    {
        return false;
    }

    public function getApiUri(): string
    {
        return self::API_URI;
    }

    public function getOAuth2UriCollection(): UriCollectionInterface
    {
        return $this->oAuth2Uris ?? $this->oAuth2Uris = new UriCollection(
            self::OAUTH2_AUTH_ENDPOINT_URI,
            self::OAUTH2_TOKEN_ENDPOINT_URI
        );
    }

    public function getGeoWidgetUri(): string
    {
        return self::GEO_WIDGET_URI;
    }
}
