<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Hook\Event\Mail;

use InPost\International\Hook\Event\AbstractActionHook;
use InPost\International\Hook\Event\ParametersUpdatingHookInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class GetExtraTemplateVariablesEvent extends AbstractActionHook implements ParametersUpdatingHookInterface
{
    public const NAME = 'actionGetExtraMailTemplateVars';

    /**
     * @var string
     */
    private $template;

    /**
     * @var array
     */
    private $variables;

    /**
     * @var array
     */
    private $extraVariables;

    /**
     * @var int
     */
    private $languageId;

    /**
     * @param array<string, mixed> $template_vars
     * @param array<string, mixed> $extra_template_vars
     */
    public function __construct(string $template, array $template_vars, array $extra_template_vars, int $id_lang)
    {
        $this->template = $template;
        $this->variables = $template_vars;
        $this->extraVariables = $extra_template_vars;
        $this->languageId = $id_lang;
    }

    public function getName(): string
    {
        return self::NAME;
    }

    public function getTemplate(): string
    {
        return $this->template;
    }

    /**
     * @return array<string, mixed>
     */
    public function getOriginalVariables(): array
    {
        return $this->variables;
    }

    public function hasOriginalVariable(string $name): bool
    {
        return array_key_exists($name, $this->variables);
    }

    /**
     * @return mixed
     */
    public function getOriginalVariable(string $name)
    {
        return $this->variables[$name] ?? null;
    }

    /**
     * @return array<string, mixed>
     */
    public function getExtraVariables(): array
    {
        return $this->extraVariables;
    }

    /**
     * @return mixed
     */
    public function getExtraVariable(string $name)
    {
        return $this->extraVariables[$name] ?? null;
    }

    /**
     * @return mixed
     */
    public function getVariable(string $name)
    {
        if (array_key_exists($name, $this->extraVariables)) {
            return $this->extraVariables[$name];
        }

        return $this->getOriginalVariable($name);
    }

    /**
     * @param mixed $value
     */
    public function setVariable(string $name, $value): void
    {
        $this->extraVariables[$name] = $value;
    }

    public function resetVariable(string $name): void
    {
        unset($this->extraVariables[$name]);
    }

    public function getLanguageId(): int
    {
        return $this->languageId;
    }

    /**
     * {@inheritDoc}
     */
    public function updateParameters(array &$parameters): void
    {
        $parameters['extra_template_vars'] = $this->extraVariables;
    }
}
