<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Installer;

use Doctrine\DBAL\Exception\TableNotFoundException;
use InPost\International\Carrier\CarrierRepository;
use InPost\International\Carrier\CarrierRepositoryInterface;
use InPost\International\Carrier\CarrierType;
use InPost\International\Carrier\Message\CreateCarrierCommand;
use InPost\International\Carrier\Message\DeleteCarrierCommand;
use InPost\International\Entity\Carrier;
use InPost\International\Installer\Exception\InstallerException;
use PrestaShop\PrestaShop\Core\CommandBus\CommandBusInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class CarrierInstaller implements InstallerInterface, UninstallerInterface
{
    /**
     * @var TranslatorInterface
     */
    private $translator;

    /**
     * @var CommandBusInterface
     */
    private $bus;

    /**
     * @var CarrierRepository
     */
    private $repository;

    /**
     * @param CarrierRepository $repository
     */
    public function __construct(TranslatorInterface $translator, CommandBusInterface $bus, CarrierRepositoryInterface $repository)
    {
        $this->translator = $translator;
        $this->bus = $bus;
        $this->repository = $repository;
    }

    public function install(\Module $module): void
    {
        if (0 !== $this->repository->count([])) {
            return;
        }

        try {
            $this->createCarrier(CarrierType::PointDelivery());
            $this->createCarrier(CarrierType::Courier());
        } catch (\Exception $e) {
            throw new InstallerException($this->translator->trans('Could not create module\'s carriers.', [], 'Modules.Inpostinternational.Installer'), 0, $e);
        }
    }

    public function uninstall(\Module $module, bool $keepData): void
    {
        if ($keepData) {
            return;
        }

        try {
            $carriers = $this->repository->findAll();
        } catch (TableNotFoundException $e) {
            return;
        }

        foreach ($carriers as $carrier) {
            try {
                $this->deleteCarrier($carrier);
            } catch (\Exception $e) {
                throw new InstallerException($this->translator->trans('Failed to delete carrier with ID "{id}".', ['{id}' => $carrier->getReferenceId()], 'Modules.Inpostinternational.Installer'), 0, $e);
            }
        }
    }

    private function createCarrier(CarrierType $type): void
    {
        $command = new CreateCarrierCommand();
        $command->setType($type);
        $command->setCountries($type->getPossibleDestinationCountries());

        $this->bus->handle($command);
    }

    private function deleteCarrier(Carrier $carrier): void
    {
        $command = new DeleteCarrierCommand($carrier->getReferenceId());

        $this->bus->handle($command);
    }
}
