<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\OAuth2\Token;

use InPost\International\Clock\SystemClock;
use InPost\International\OAuth2\Exception\UnexpectedValueException;
use Psr\Clock\ClockInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * @mixin AccessTokenFactoryInterface
 */
trait AccessTokenFactoryTrait
{
    /**
     * @var ClockInterface
     */
    private $clock;

    /**
     * @var int|null
     */
    private $defaultExpirationTime;

    public function __construct(ClockInterface $clock = null, int $defaultExpirationTime = null)
    {
        $this->clock = $clock ?? SystemClock::fromSystemTimezone();
        $this->defaultExpirationTime = $defaultExpirationTime;
    }

    private function getExpiresAt(array $data): ?\DateTimeImmutable
    {
        $expiresIn = isset($data['expires_in']) ? (int) $data['expires_in'] : $this->defaultExpirationTime;

        if (null === $expiresIn) {
            return null;
        }

        if (0 > $expiresIn) {
            throw new UnexpectedValueException('Negative access token expiration time.');
        }

        return $this->clock->now()->add(new \DateInterval(sprintf('PT%dS', $expiresIn)));
    }

    private function getScopes(array $data): ?array
    {
        if (!isset($data['scope'])) {
            return null;
        }

        return explode(' ', $data['scope']);
    }
}
