<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Order\EventListener\Admin;

use InPost\International\Configuration\Repository\ApiConfigurationRepositoryInterface;
use InPost\International\Controller\Admin\ShipmentController;
use InPost\International\Order\Event\Admin\DisplayOrderTabContentEvent;
use InPost\International\Order\Event\Admin\DisplayOrderTabLinkEvent;
use InPost\International\Shipment\Grid\ShipmentFilters;
use PrestaShop\PrestaShop\Core\Grid\GridFactoryInterface;
use PrestaShop\PrestaShop\Core\Grid\Presenter\GridPresenterInterface;
use PrestaShopBundle\Security\Voter\PageVoter;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Security\Core\Authorization\AuthorizationCheckerInterface;
use Twig\Environment;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class DisplayShipmentsListListener implements EventSubscriberInterface
{
    /**
     * @var AuthorizationCheckerInterface
     */
    private $authorizationChecker;

    /**
     * @var Environment
     */
    private $twig;

    /**
     * @var ApiConfigurationRepositoryInterface
     */
    private $configuration;

    /**
     * @var GridFactoryInterface
     */
    private $gridFactory;

    /**
     * @var GridPresenterInterface
     */
    private $gridPresenter;

    public function __construct(AuthorizationCheckerInterface $authorizationChecker, Environment $twig, ApiConfigurationRepositoryInterface $configuration, GridFactoryInterface $gridFactory, GridPresenterInterface $gridPresenter)
    {
        $this->authorizationChecker = $authorizationChecker;
        $this->twig = $twig;
        $this->configuration = $configuration;
        $this->gridFactory = $gridFactory;
        $this->gridPresenter = $gridPresenter;
    }

    public static function getSubscribedEvents(): array
    {
        return [
            DisplayOrderTabLinkEvent::NAME => 'onDisplayTabLink',
            DisplayOrderTabContentEvent::NAME => 'onDisplayTabContent',
        ];
    }

    public function onDisplayTabLink(DisplayOrderTabLinkEvent $event): void
    {
        if (!$this->authorizationChecker->isGranted(PageVoter::READ, ShipmentController::TAB_NAME)) {
            return;
        }

        $content = $this->twig->render('@Modules/inpostinternational/views/templates/hook/admin/orders/shipments_tab_link.html.twig');
        $event->appendContent($content);
    }

    public function onDisplayTabContent(DisplayOrderTabContentEvent $event): void
    {
        if (!$this->authorizationChecker->isGranted(PageVoter::READ, ShipmentController::TAB_NAME)) {
            return;
        }

        $filters = ShipmentFilters::buildDefaults();
        $filters->addFilter([
            'id_order' => $orderId = $event->getOrderId(),
            'is_sandbox' => $this->configuration->getConfiguration()->getEnvironment()->isTestEnvironment(),
        ]);
        $filters->set('limit', null);

        $grid = $this->gridFactory->getGrid($filters);

        $content = $this->twig->render('@Modules/inpostinternational/views/templates/hook/admin/orders/shipments_tab_content.html.twig', [
            'can_create' => $this->authorizationChecker->isGranted(PageVoter::CREATE, ShipmentController::TAB_NAME),
            'order_id' => $orderId,
            'grid' => $this->gridPresenter->present($grid),
        ]);

        $event->appendContent($content);
    }
}
