<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Order\EventListener;

use InPost\International\Configuration\Repository\MailsConfigurationRepositoryInterface;
use InPost\International\Hook\Event\Mail\GetExtraTemplateVariablesEvent;
use InPost\International\Order\Util\MailHelper;
use Psr\Container\ContainerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Contracts\Service\ServiceSubscriberInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class ModifyOrderNotificationsListener implements EventSubscriberInterface, ServiceSubscriberInterface
{
    /**
     * @var MailsConfigurationRepositoryInterface
     */
    private $configuration;

    /**
     * @var ContainerInterface
     */
    private $container;

    public function __construct(MailsConfigurationRepositoryInterface $configuration, ContainerInterface $container)
    {
        $this->configuration = $configuration;
        $this->container = $container;
    }

    public static function getSubscribedEvents(): array
    {
        return [
            GetExtraTemplateVariablesEvent::NAME => 'modifyVariables',
        ];
    }

    public static function getSubscribedServices(): array
    {
        return [
            MailHelper::class,
        ];
    }

    public function modifyVariables(GetExtraTemplateVariablesEvent $event): void
    {
        switch ($event->getTemplate()) {
            case 'order_conf':
                $this->modifyOrderConfirmationVars($event);
                break;
            case 'new_order':
                $this->modifyNewOrderNotificationVars($event);
                break;
            default:
                break;
        }
    }

    private function modifyOrderConfirmationVars(GetExtraTemplateVariablesEvent $event): void
    {
        if (!$this->configuration->getMailsConfiguration()->modifyOrderConfirmation()) {
            return;
        }

        if (0 >= $orderId = (int) $event->getOriginalVariable('{id_order}')) {
            return;
        }

        if (null === $carrierName = $event->getVariable('{carrier}')) {
            return;
        }

        $pointName = $this->getHelper()->getPointNameByOrderId($orderId, $event->getLanguageId());

        if (null === $pointName) {
            return;
        }

        $event->setVariable('{carrier}', sprintf('%s (%s)', $carrierName, $pointName));
    }

    private function modifyNewOrderNotificationVars(GetExtraTemplateVariablesEvent $event): void
    {
        if ('' === $orderReference = (string) $event->getOriginalVariable('{order_name}')) {
            return;
        }

        if (!$event->hasOriginalVariable('{delivery_block_html}')) {
            return;
        }

        $address = $this->getHelper()->getPointAddressByOrderReference($orderReference, $event->getLanguageId());

        if (null === $address) {
            return;
        }

        $event->setVariable('{delivery_block_txt}', $address);
        $event->setVariable('{delivery_block_html}', nl2br($address));
    }

    private function getHelper(): MailHelper
    {
        return $this->container->get(MailHelper::class);
    }
}
