<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Order\Grid;

use InPost\International\PrestaShop\Grid\Column\DataColumn as DataColumnWithAttr;
use InPost\International\PrestaShop\Grid\Column\LinkColumn;
use PrestaShop\PrestaShop\Core\Grid\Action\GridActionCollection;
use PrestaShop\PrestaShop\Core\Grid\Action\GridActionCollectionInterface;
use PrestaShop\PrestaShop\Core\Grid\Action\Row\RowActionCollection;
use PrestaShop\PrestaShop\Core\Grid\Action\Row\RowActionCollectionInterface;
use PrestaShop\PrestaShop\Core\Grid\Action\Row\Type\LinkRowAction;
use PrestaShop\PrestaShop\Core\Grid\Action\Row\Type\SubmitRowAction;
use PrestaShop\PrestaShop\Core\Grid\Action\Type\LinkGridAction;
use PrestaShop\PrestaShop\Core\Grid\Action\Type\SimpleGridAction;
use PrestaShop\PrestaShop\Core\Grid\Column\ColumnCollection;
use PrestaShop\PrestaShop\Core\Grid\Column\ColumnCollectionInterface;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\ActionColumn;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\DataColumn;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\DateTimeColumn;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\IdentifierColumn;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\DataColumn as LegacyDataColumn;
use PrestaShop\PrestaShop\Core\Grid\Definition\Factory\AbstractGridDefinitionFactory;
use PrestaShop\PrestaShop\Core\Hook\HookDispatcherInterface;
use PrestaShopBundle\Event\Dispatcher\NullDispatcher;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Contracts\Translation\TranslatorInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class OrderShipmentGridDefinitionFactory extends AbstractGridDefinitionFactory
{
    public const GRID_ID = 'inpost_intl_order_shipment';

    /**
     * @var RequestStack
     */
    private $requestStack;

    public function __construct(TranslatorInterface $translator, RequestStack $requestStack, HookDispatcherInterface $hookDispatcher = null)
    {
        parent::__construct($hookDispatcher ?? new NullDispatcher());
        $this->translator = $translator;
        $this->requestStack = $requestStack;
    }

    protected function getId(): string
    {
        return self::GRID_ID;
    }

    protected function getName(): string
    {
        return $this->trans('Shipments', [], 'Modules.Inpostinternational.Shipment');
    }

    protected function getColumns(): ColumnCollectionInterface
    {
        $dataColumnClass = class_exists(DataColumn::class) ? DataColumn::class : LegacyDataColumn::class;

        return (new ColumnCollection())
            ->add(
                (new IdentifierColumn('id'))
                    ->setName($this->trans('ID', [], 'Admin.Global'))
                    ->setOptions([
                        'identifier_field' => 'id',
                        'sortable' => false,
                    ])
            )
            ->add(
                (new LinkColumn('tracking_number'))
                    ->setName($this->trans('Shipment number', [], 'Modules.Inpostinternational.Shipment'))
                    ->setOptions([
                        'field' => 'tracking_number',
                        'sortable' => false,
                        'route' => 'admin_inpost_intl_shipments_tracking',
                        'route_param_name' => 'id',
                        'route_param_field' => 'id',
                        'attr' => [
                            'class' => 'js-inpost-intl-parcel-tracking-link',
                        ],
                    ])
            )
            ->add(
                (new DataColumnWithAttr('status'))
                    ->setName($this->trans('Status', [], 'Modules.Inpostinternational.Shipment'))
                    ->setOptions([
                        'field' => 'status',
                        'attr' => [
                            'class' => 'js-inpost-intl-shipment-status',
                        ],
                    ])
            )
            ->add(
                (new $dataColumnClass('parcel_comment'))
                    ->setName($this->trans('Customer reference', [], 'Modules.Inpostinternational.Shipment'))
                    ->setOptions([
                        'field' => 'parcel_comment',
                        'sortable' => false,
                    ])
            )
            ->add(
                (new DateTimeColumn('created_at'))
                    ->setName($this->trans('Created at', [], 'Modules.Inpostinternational.Admin'))
                    ->setOptions([
                        'field' => 'created_at',
                        'format' => 'Y-m-d H:i:s',
                        'sortable' => false,
                    ])
            )
            ->add(
                (new ActionColumn('actions'))
                    ->setName($this->trans('Actions', [], 'Admin.Global'))
                    ->setOptions([
                        'actions' => $this->getRowActions(),
                    ])
            );
    }

    protected function getGridActions(): GridActionCollectionInterface
    {
        return (new GridActionCollection())
            ->add(
                (new LinkGridAction('refresh_list'))
                    ->setName($this->trans('Refresh list', [], 'Admin.Advparameters.Feature'))
                    ->setOptions([
                        'route' => 'admin_inpost_intl_orders_shipments',
                        'route_params' => [
                            'orderIds' => $this->requestStack->getCurrentRequest()->attributes->getInt('orderId'),
                        ],
                    ])
                    ->setIcon('refresh')
            )
            ->add(
                (new SimpleGridAction('common_show_query'))
                    ->setName($this->trans('Show SQL query', [], 'Admin.Actions'))
                    ->setIcon('code')
            );
    }

    private function getRowActions(): RowActionCollectionInterface
    {
        return (new RowActionCollection())
            ->add(
                (new LinkRowAction('print_label'))
                    ->setName($this->trans('Print label', [], 'Modules.Inpostinternational.Shipment'))
                    ->setOptions([
                        'route' => 'admin_inpost_intl_shipments_print_label',
                        'route_param_name' => 'id',
                        'route_param_field' => 'id',
                    ])
                    ->setIcon('print')
            )
            ->add(
                (new SubmitRowAction('update_status'))
                    ->setName($this->trans('Update status', [], 'Modules.Inpostinternational.Shipment'))
                    ->setOptions([
                        'route' => 'admin_inpost_intl_shipments_update_status',
                        'route_param_name' => 'id',
                        'route_param_field' => 'id',
                    ])
                    ->setIcon('refresh')
            );
    }
}
