<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Order\View;

use Doctrine\ORM\EntityRepository;
use InPost\International\Delivery\Point\PointDataProviderInterface;
use InPost\International\Delivery\Point\Presenter\PointPresenterInterface;
use InPost\International\Order\Util\OrderHelper;
use InPost\International\Order\View\Model\DeliveryPointView;
use PrestaShopBundle\Entity\Lang;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class DeliveryPointViewFactory
{
    /**
     * @var OrderHelper
     */
    private $orderHelper;

    /**
     * @var PointDataProviderInterface
     */
    private $pointDataProvider;

    /**
     * @var PointPresenterInterface
     */
    private $addressPresenter;

    /**
     * @var EntityRepository<Lang>|null
     */
    private $languageRepository;

    public function __construct(OrderHelper $orderHelper, PointDataProviderInterface $pointDataProvider, PointPresenterInterface $addressPresenter)
    {
        $this->orderHelper = $orderHelper;
        $this->pointDataProvider = $pointDataProvider;
        $this->addressPresenter = $addressPresenter;
    }

    /**
     * @param EntityRepository<Lang> $languageRepository
     *
     * @internal
     *
     * @deprecated to be removed when we drop support for early PS 1.7.7 versions
     */
    public function setLanguageRepository(EntityRepository $languageRepository): void
    {
        $this->languageRepository = $languageRepository;
    }

    public function getView(\Order $order): ?DeliveryPointView
    {
        if (null === $pointId = $this->orderHelper->getDeliveryPointId($order)) {
            return null;
        }

        $view = new DeliveryPointView($pointId);

        if (null !== $address = $this->getPointAddress($pointId, $order)) {
            $view->setFormattedAddress(nl2br($address));
        }

        return $view;
    }

    private function getPointAddress(string $pointId, \Order $order): ?string
    {
        try {
            $point = $this->pointDataProvider->getPoint($pointId);
        } catch (\Exception $e) {
            return null;
        }

        if (null === $point) {
            return null;
        }

        $locale = $this->getLocale($order);

        return $this->addressPresenter->formatAddress($point, $locale);
    }

    private function getLocale(\Order $order): ?string
    {
        // method was added in PS 1.7.7.3
        if (is_callable([$order, 'getAssociatedLanguage'])) {
            return $order->getAssociatedLanguage()->locale;
        }

        if (!isset($this->languageRepository)) {
            return null;
        }

        if (null === $language = $this->languageRepository->find((int) $order->id_lang)) {
            return null;
        }

        return $language->getLocale();
    }
}
