<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\ParcelTemplate\Grid;

use InPost\International\Api\Shipment\Model\Parcel\DimensionUnit;
use InPost\International\Api\Shipment\Model\Parcel\ParcelType;
use InPost\International\Api\Shipment\Model\Parcel\WeightUnit;
use PrestaShop\PrestaShop\Core\Grid\Data\Factory\GridDataFactoryInterface;
use PrestaShop\PrestaShop\Core\Grid\Data\GridData;
use PrestaShop\PrestaShop\Core\Grid\Data\GridDataInterface;
use PrestaShop\PrestaShop\Core\Grid\Record\RecordCollection;
use PrestaShop\PrestaShop\Core\Grid\Search\SearchCriteriaInterface;
use PrestaShop\PrestaShop\Core\Localization\LocaleInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class ParcelTemplateGridDataFactory implements GridDataFactoryInterface
{
    /**
     * @var GridDataFactoryInterface
     */
    private $dataFactory;

    /**
     * @var TranslatorInterface
     */
    private $translator;

    /**
     * @var LocaleInterface
     */
    private $locale;

    public function __construct(GridDataFactoryInterface $dataFactory, TranslatorInterface $translator, LocaleInterface $locale)
    {
        $this->dataFactory = $dataFactory;
        $this->translator = $translator;
        $this->locale = $locale;
    }

    public function getData(SearchCriteriaInterface $searchCriteria): GridDataInterface
    {
        $data = $this->dataFactory->getData($searchCriteria);

        $records = array_map([$this, 'modifyRecord'], $data->getRecords()->all());

        return new GridData(
            new RecordCollection($records),
            $data->getRecordsTotal(),
            $data->getQuery()
        );
    }

    private function modifyRecord(array $record): array
    {
        $record['type'] = ParcelType::from($record['type'])->trans($this->translator);
        $record['dimensions'] = $this->formatDimensions($record);
        $record['weight'] = $this->formatWeight($record);

        return $record;
    }

    private function formatDimensions(array $record): string
    {
        $unit = DimensionUnit::from($record['length_unit']);

        return sprintf(
            '%sｘ%sｘ%s %s',
            $this->formatNumber($record['length']),
            $this->formatNumber($record['width']),
            $this->formatNumber($record['height']),
            $unit->getSymbol()
        );
    }

    private function formatWeight(array $record): string
    {
        $unit = WeightUnit::from($record['weight_unit']);

        return sprintf(
            '%s %s',
            $this->formatNumber($record['weight']),
            $unit->getSymbol()
        );
    }

    /**
     * @param float|numeric-string $number
     */
    private function formatNumber($number): string
    {
        return $this->locale->formatNumber($number);
    }
}
