<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\ParcelTemplate\Grid;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\ParameterType;
use Doctrine\DBAL\Query\QueryBuilder;
use PrestaShop\PrestaShop\Core\Grid\Query\AbstractDoctrineQueryBuilder;
use PrestaShop\PrestaShop\Core\Grid\Query\DoctrineSearchCriteriaApplicatorInterface;
use PrestaShop\PrestaShop\Core\Grid\Search\SearchCriteriaInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class ParcelTemplateQueryBuilder extends AbstractDoctrineQueryBuilder
{
    /**
     * @var DoctrineSearchCriteriaApplicatorInterface
     */
    private $searchCriteriaApplicator;

    public function __construct(Connection $connection, string $dbPrefix, DoctrineSearchCriteriaApplicatorInterface $searchCriteriaApplicator)
    {
        parent::__construct($connection, $dbPrefix);
        $this->searchCriteriaApplicator = $searchCriteriaApplicator;
    }

    public function getSearchQueryBuilder(SearchCriteriaInterface $searchCriteria): QueryBuilder
    {
        $queryBuilder = $this
            ->getCommonQueryBuilder($searchCriteria)
            ->select('t.*');

        $this->searchCriteriaApplicator->applyPagination($searchCriteria, $queryBuilder);
        $this->applySorting($searchCriteria, $queryBuilder);

        return $queryBuilder;
    }

    public function getCountQueryBuilder(SearchCriteriaInterface $searchCriteria): QueryBuilder
    {
        return $this
            ->getCommonQueryBuilder($searchCriteria)
            ->select('COUNT(*)');
    }

    private function getCommonQueryBuilder(SearchCriteriaInterface $searchCriteria): QueryBuilder
    {
        $queryBuilder = $this->connection
            ->createQueryBuilder()
            ->from($this->dbPrefix . 'inpost_intl_parcel_template', 't');

        $this->applyFilters($searchCriteria->getFilters(), $queryBuilder);

        return $queryBuilder;
    }

    /**
     * @param array<string, mixed> $filters
     */
    private function applyFilters(array $filters, QueryBuilder $qb): void
    {
        foreach ($filters as $filterName => $filterValue) {
            switch ($filterName) {
                case 'id':
                case 'type':
                case 'weight':
                    $qb->andWhere('t.' . $filterName . ' = :' . $filterName);
                    $qb->setParameter($filterName, $filterValue);

                    break;
                case 'name':
                    $qb->andWhere('t.name LIKE :name');
                    $qb->setParameter('name', '%' . $filterValue . '%');

                    break;
                case 'is_default':
                    $qb->andWhere('t.is_default = :default');
                    $qb->setParameter('default', $filterValue, ParameterType::BOOLEAN);

                    break;
                case 'dimensions':
                    $qb->andWhere('t.length = :dimensions OR t.width = :dimensions OR t.height = :dimensions');
                    $qb->setParameter('dimensions', $filterValue);

                    break;
                default:
                    break;
            }
        }
    }

    private function applySorting(SearchCriteriaInterface $searchCriteria, QueryBuilder $qb): void
    {
        if ('dimensions' === $searchCriteria->getOrderBy()) {
            $qb->addOrderBy('(t.length * t.width * t.height)', $searchCriteria->getOrderWay());
        } else {
            $this->searchCriteriaApplicator->applySorting($searchCriteria, $qb);
        }
    }
}
