<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\ParcelTemplate\Message;

use InPost\International\Api\Shipment\Model\Parcel\ParcelType;
use InPost\International\Entity\ParcelTemplate;
use InPost\International\ParcelTemplate\Exception\ParcelTemplateException;
use InPost\International\Shipment\DTO\Dimensions;
use InPost\International\Shipment\DTO\Weight;
use Symfony\Component\Validator\Constraints as Assert;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class EditParcelTemplateCommand
{
    /**
     * @var int
     *
     * @Assert\GreaterThan(0, message="Invalid template ID.")
     */
    private $templateId;

    /**
     * @var string|null
     *
     * @Assert\Length(max=64)
     */
    private $name;

    /**
     * @var ParcelType|null
     */
    private $type;

    /**
     * @var Dimensions|null
     *
     * @Assert\Valid
     */
    private $dimensions;

    /**
     * @var Weight|null
     *
     * @Assert\Valid
     */
    private $weight;

    /**
     * @var bool|null
     */
    private $default = false;

    public function __construct(int $addressId)
    {
        $this->templateId = $addressId;
    }

    public static function for(ParcelTemplate $template): self
    {
        if (null === $id = $template->getId()) {
            throw new ParcelTemplateException('Template has not been persisted yet.');
        }

        $command = new self($id);

        $dimensions = $template->getDimensions();
        $dimensionsDto = (new Dimensions())
            ->setLength($dimensions->getLength())
            ->setWidth($dimensions->getWidth())
            ->setHeight($dimensions->getHeight())
            ->setUnit($dimensions->getUnit());

        $weight = $template->getWeight();
        $weightDto = (new Weight())
            ->setAmount($weight->getWeight())
            ->setUnit($weight->getUnit());

        $command->name = $template->getName();
        $command->type = $template->getType();
        $command->dimensions = $dimensionsDto;
        $command->weight = $weightDto;
        $command->default = $template->isDefault();

        return $command;
    }

    public function getTemplateId(): int
    {
        return $this->templateId;
    }

    public function getName(): ?string
    {
        return $this->name;
    }

    public function setName(?string $name): void
    {
        $this->name = $name;
    }

    public function getType(): ?ParcelType
    {
        return $this->type;
    }

    public function setType(?ParcelType $type): void
    {
        $this->type = $type;
    }

    public function getDimensions(): ?Dimensions
    {
        return $this->dimensions;
    }

    public function setDimensions(?Dimensions $dimensions): void
    {
        $this->dimensions = $dimensions;
    }

    public function getWeight(): ?Weight
    {
        return $this->weight;
    }

    public function setWeight(?Weight $weight): void
    {
        $this->weight = $weight;
    }

    public function isDefault(): ?bool
    {
        return $this->default;
    }

    public function setDefault(?bool $default): void
    {
        $this->default = $default;
    }
}
