<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\PickupAddress\Message;

use InPost\International\Common\DTO\Phone;
use InPost\International\Entity\PickupAddress;
use InPost\International\PickupAddress\DTO\Address;
use InPost\International\PickupAddress\DTO\ContactPerson;
use InPost\International\PickupAddress\Exception\PickupAddressException;
use Symfony\Component\Validator\Constraints as Assert;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class EditPickupAddressCommand
{
    /**
     * @var int
     *
     * @Assert\GreaterThan(0, message="Invalid address ID.")
     */
    private $addressId;

    /**
     * @var string|null
     *
     * @Assert\Length(max=64)
     */
    private $name;

    /**
     * @var ContactPerson|null
     *
     * @Assert\Valid
     */
    private $contactPerson;

    /**
     * @var Address|null
     *
     * @Assert\Valid
     */
    private $address;

    /**
     * @var bool|null
     *
     * @Assert\NotNull
     */
    private $default = false;

    public function __construct(int $addressId)
    {
        $this->addressId = $addressId;
    }

    public static function for(PickupAddress $pickupAddress): self
    {
        if (null === $id = $pickupAddress->getId()) {
            throw new PickupAddressException('Address has not been persisted yet.');
        }

        $command = new self($id);

        $contactPerson = $pickupAddress->getContactPerson();
        $phone = $contactPerson->getPhone();

        $phoneDto = (new Phone())
            ->setPrefix($phone->getPrefix())
            ->setNumber($phone->getNumber());

        $contactPersonDto = (new ContactPerson())
            ->setFirstName($contactPerson->getFirstName())
            ->setLastName($contactPerson->getLastName())
            ->setEmail($contactPerson->getEmail())
            ->setPhone($phoneDto);

        $address = $pickupAddress->getAddress();
        $addressDto = (new Address())
            ->setStreet($address->getStreet())
            ->setHouseNumber($address->getHouseNumber())
            ->setFlatNumber($address->getFlatNumber())
            ->setPostcode($address->getPostcode())
            ->setCity($address->getCity())
            ->setCountry($address->getCountry())
            ->setDescription($address->getDescription());

        $command->name = $pickupAddress->getName();
        $command->contactPerson = $contactPersonDto;
        $command->address = $addressDto;
        $command->default = $pickupAddress->isDefault();

        return $command;
    }

    public function getAddressId(): int
    {
        return $this->addressId;
    }

    public function getName(): ?string
    {
        return $this->name;
    }

    public function setName(?string $name): void
    {
        $this->name = $name;
    }

    public function getContactPerson(): ?ContactPerson
    {
        return $this->contactPerson;
    }

    public function setContactPerson(?ContactPerson $contactPerson): void
    {
        $this->contactPerson = $contactPerson;
    }

    public function getAddress(): ?Address
    {
        return $this->address;
    }

    public function setAddress(?Address $address): void
    {
        $this->address = $address;
    }

    public function isDefault(): ?bool
    {
        return $this->default;
    }

    public function setDefault(?bool $default): void
    {
        $this->default = $default;
    }
}
