<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\PickupAddress;

use InPost\International\Common\DTO\Phone as PhoneDto;
use InPost\International\Entity\Address;
use InPost\International\Entity\ContactPerson;
use InPost\International\Entity\Phone;
use InPost\International\PickupAddress\DTO\Address as AddressDto;
use InPost\International\PickupAddress\DTO\ContactPerson as ContactPersonDto;
use InPost\International\PickupAddress\Exception\PickupAddressException;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class PickupAddressMapper
{
    public function mapDtoToContactPerson(ContactPersonDto $contactPerson): ContactPerson
    {
        if (null === $phoneDto = $contactPerson->getPhone()) {
            throw new PickupAddressException('Phone number is required.');
        }

        return new ContactPerson(
            (string) $contactPerson->getFirstName(),
            (string) $contactPerson->getLastName(),
            (string) $contactPerson->getEmail(),
            $this->mapDtoToPhone($phoneDto)
        );
    }

    public function mapDtoToAddress(AddressDto $address): Address
    {
        if (null === $country = $address->getCountry()) {
            throw new PickupAddressException('Country is required.');
        }

        return new Address(
            (string) $address->getStreet(),
            (string) $address->getHouseNumber(),
            (string) $address->getPostcode(),
            (string) $address->getCity(),
            $country,
            $address->getFlatNumber(),
            $address->getDescription()
        );
    }

    public function mapDtoToPhone(PhoneDto $phone): Phone
    {
        return new Phone(
            (string) $phone->getPrefix(),
            (string) $phone->getNumber()
        );
    }
}
