<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\PickupOrder\Grid;

use InPost\International\PrestaShop\Grid\Column\ListColumn;
use PrestaShop\PrestaShop\Core\Grid\Action\GridActionCollection;
use PrestaShop\PrestaShop\Core\Grid\Action\GridActionCollectionInterface;
use PrestaShop\PrestaShop\Core\Grid\Action\Type\SimpleGridAction;
use PrestaShop\PrestaShop\Core\Grid\Column\ColumnCollection;
use PrestaShop\PrestaShop\Core\Grid\Column\ColumnCollectionInterface;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\ActionColumn;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\DataColumn;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\DateTimeColumn;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\Common\IdentifierColumn;
use PrestaShop\PrestaShop\Core\Grid\Column\Type\DataColumn as LegacyDataColumn;
use PrestaShop\PrestaShop\Core\Grid\Definition\Factory\AbstractFilterableGridDefinitionFactory;
use PrestaShop\PrestaShop\Core\Grid\Filter\Filter;
use PrestaShop\PrestaShop\Core\Grid\Filter\FilterCollection;
use PrestaShop\PrestaShop\Core\Grid\Filter\FilterCollectionInterface;
use PrestaShop\PrestaShop\Core\Hook\HookDispatcherInterface;
use PrestaShopBundle\Event\Dispatcher\NullDispatcher;
use PrestaShopBundle\Form\Admin\Type\DateRangeType;
use PrestaShopBundle\Form\Admin\Type\SearchAndResetType;
use Symfony\Component\Form\Extension\Core\Type\NumberType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Contracts\Translation\TranslatorInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class PickupOrderGridDefinitionFactory extends AbstractFilterableGridDefinitionFactory
{
    public const GRID_ID = 'inpost_intl_pickup_order';

    public function __construct(TranslatorInterface $translator, HookDispatcherInterface $hookDispatcher = null)
    {
        parent::__construct($hookDispatcher ?? new NullDispatcher());
        $this->translator = $translator;
    }

    protected function getId(): string
    {
        return self::GRID_ID;
    }

    protected function getName(): string
    {
        return $this->trans('Pickup orders', [], 'Modules.Inpostinternational.Pickup');
    }

    protected function getColumns(): ColumnCollectionInterface
    {
        $dataColumnClass = class_exists(DataColumn::class) ? DataColumn::class : LegacyDataColumn::class;

        return (new ColumnCollection())
            ->add(
                (new IdentifierColumn('id'))
                    ->setName($this->trans('ID', [], 'Admin.Global'))
                    ->setOptions([
                        'identifier_field' => 'id',
                    ])
            )
            ->add(
                (new $dataColumnClass('tracking_number'))
                    ->setName($this->trans('Tracking number', [], 'Modules.Inpostinternational.Pickup'))
                    ->setOptions([
                        'field' => 'tracking_number',
                    ])
            )
            ->add(
                (new $dataColumnClass('pickup_address'))
                    ->setName($this->trans('Pickup address', [], 'Modules.Inpostinternational.Pickup'))
                    ->setOptions([
                        'field' => 'pickup_address',
                    ])
            )
            ->add(
                (new $dataColumnClass('pickup_time'))
                    ->setName($this->trans('Pickup time', [], 'Modules.Inpostinternational.Pickup'))
                    ->setOptions([
                        'field' => 'pickup_time',
                        'sortable' => false,
                    ])
            )
            ->add(
                (new ListColumn('shipment_numbers'))
                    ->setName($this->trans('Shipments', [], 'Modules.Inpostinternational.Pickup'))
                    ->setOptions([
                        'field' => 'shipment_numbers',
                        'sortable' => false,
                    ])
            )
            ->add(
                (new DateTimeColumn('created_at'))
                    ->setName($this->trans('Created at', [], 'Modules.Inpostinternational.Admin'))
                    ->setOptions([
                        'field' => 'created_at',
                    ])
            )
            ->add(new ActionColumn('actions'));
    }

    protected function getFilters(): FilterCollectionInterface
    {
        return (new FilterCollection())
            ->add(
                (new Filter('id', NumberType::class))
                    ->setTypeOptions([
                        'attr' => [
                            'placeholder' => $this->trans('Search ID', [], 'Admin.Actions'),
                        ],
                        'required' => false,
                    ])
                    ->setAssociatedColumn('id')
            )
            ->add(
                (new Filter('tracking_number', TextType::class))
                    ->setTypeOptions([
                        'attr' => [
                            'placeholder' => $this->trans('Search tracking number', [], 'Modules.Inpostinternational.Pickup'),
                        ],
                        'required' => false,
                    ])
                    ->setAssociatedColumn('tracking_number')
            )
            ->add(
                (new Filter('shipment_number', TextType::class))
                    ->setTypeOptions([
                        'attr' => [
                            'placeholder' => $this->trans('Search shipments', [], 'Modules.Inpostinternational.Pickup'),
                        ],
                        'required' => false,
                    ])
                    ->setAssociatedColumn('shipment_numbers')
            )
            ->add(
                (new Filter('created_at', DateRangeType::class))
                    ->setTypeOptions([
                        'required' => false,
                    ])
                    ->setAssociatedColumn('created_at')
            )
            ->add(
                (new Filter('actions', SearchAndResetType::class))
                    ->setTypeOptions([
                        'reset_route' => 'admin_common_reset_search_by_filter_id',
                        'reset_route_params' => [
                            'filterId' => self::GRID_ID,
                        ],
                        'redirect_route' => 'admin_inpost_intl_pickup_orders_index',
                    ])
                    ->setAssociatedColumn('actions')
            );
    }

    protected function getGridActions(): GridActionCollectionInterface
    {
        return (new GridActionCollection())
            ->add(
                (new SimpleGridAction('common_refresh_list'))
                    ->setName($this->trans('Refresh list', [], 'Admin.Advparameters.Feature'))
                    ->setIcon('refresh')
            )
            ->add(
                (new SimpleGridAction('common_show_query'))
                    ->setName($this->trans('Show SQL query', [], 'Admin.Actions'))
                    ->setIcon('code')
            )
            ->add(
                (new SimpleGridAction('common_export_sql_manager'))
                    ->setName($this->trans('Export to SQL Manager', [], 'Admin.Actions'))
                    ->setIcon('storage')
            );
    }
}
