<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\PickupOrder\Message;

use InPost\International\Common\DTO\Reference;
use InPost\International\Entity\PickupAddress;
use InPost\International\PickupOrder\DTO\PickupTime;
use Symfony\Component\Validator\Constraints as Assert;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class CreatePickupOrderCommand
{
    /**
     * @var int[]
     *
     * @Assert\Count(min=1)
     * @Assert\All({
     *      @Assert\Type("int"),
     *      @Assert\GreaterThan(0)
     *  })
     */
    private $shipmentIds = [];

    /**
     * @var PickupAddress|null
     *
     * @Assert\NotBlank
     */
    private $pickupAddress;

    /**
     * @var \DateTimeImmutable|null
     *
     * @Assert\NotBlank
     */
    private $pickupDate;

    /**
     * @var PickupTime|null
     *
     * @Assert\NotBlank
     * @Assert\Valid
     */
    private $pickupTime;

    /**
     * @var Reference[]
     *
     * @Assert\Count(max=3)
     */
    private $references = [];

    /**
     * @return int[]
     */
    public function getShipmentIds(): array
    {
        return $this->shipmentIds;
    }

    /**
     * @param int[] $shipmentIds
     */
    public function setShipmentIds(array $shipmentIds): self
    {
        $this->shipmentIds = $shipmentIds;

        return $this;
    }

    public function getPickupAddress(): ?PickupAddress
    {
        return $this->pickupAddress;
    }

    public function setPickupAddress(?PickupAddress $pickupAddress): self
    {
        $this->pickupAddress = $pickupAddress;

        return $this;
    }

    public function getPickupDate(): ?\DateTimeImmutable
    {
        return $this->pickupDate;
    }

    public function setPickupDate(?\DateTimeImmutable $pickupDate): self
    {
        $this->pickupDate = $pickupDate;

        return $this;
    }

    public function getPickupTime(): ?PickupTime
    {
        return $this->pickupTime;
    }

    public function setPickupTime(?PickupTime $pickupTime): self
    {
        $this->pickupTime = $pickupTime;

        return $this;
    }

    /**
     * @return Reference[]
     */
    public function getReferences(): array
    {
        return $this->references;
    }

    public function addReference(Reference $reference): self
    {
        $this->references[] = $reference;

        return $this;
    }

    public function removeReference(Reference $reference): self
    {
        if ($key = array_search($reference, $this->references, true)) {
            unset($this->references[$key]);
        }

        return $this;
    }
}
