<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\PrestaShop\ObjectModel\Form\ChoiceLoader;

use InPost\International\PrestaShop\ObjectModel\ObjectManagerInterface;
use InPost\International\PrestaShop\ObjectModel\OrderMaintainingLoaderTrait;
use InPost\International\PrestaShop\ObjectModel\QueryBuilder;
use Symfony\Component\Form\ChoiceList\ArrayChoiceList;
use Symfony\Component\Form\ChoiceList\ChoiceListInterface;
use Symfony\Component\Form\ChoiceList\Loader\ChoiceLoaderInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * @template T of \ObjectModel
 */
final class ObjectModelChoiceLoader implements ChoiceLoaderInterface
{
    use OrderMaintainingLoaderTrait;

    /**
     * @var class-string<T>
     */
    private $class;

    /**
     * @var QueryBuilder<T>|null
     */
    private $queryBuilder;

    /**
     * @var int|null
     */
    private $languageId;

    /**
     * @var int|null
     */
    private $shopId;

    /**
     * @var T[]|null
     */
    private $choices;

    /**
     * @param class-string<T> $class
     * @param QueryBuilder<T>|null $queryBuilder
     */
    public function __construct(ObjectManagerInterface $manager, string $class, QueryBuilder $queryBuilder = null, int $languageId = null, int $shopId = null)
    {
        $this->shopId = $shopId;
        $this->languageId = $languageId;
        $this->queryBuilder = $queryBuilder;
        $this->class = $class;
        $this->manager = $manager;
    }

    /**
     * {@inheritdoc}
     */
    public function loadChoiceList($value = null): ChoiceListInterface
    {
        if (!isset($this->choices)) {
            $this->choices = $this->doLoadChoices();
        }

        return new ArrayChoiceList($this->choices, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function loadValuesForChoices(array $choices, $value = null): array
    {
        if ([] === $choices) {
            return [];
        }

        if (null !== $value || isset($this->choices)) {
            return $this->loadChoiceList($value)->getValuesForChoices($choices);
        }

        $values = [];

        foreach ($choices as $i => $choice) {
            if ($choice instanceof $this->class) {
                $values[$i] = (string) $choice->id;
            } elseif (is_int($choice)) {
                $values[$i] = (string) $choice;
            }
        }

        return $values;
    }

    /**
     * {@inheritdoc}
     *
     * @param array<int, int|string> $values
     */
    public function loadChoicesForValues(array $values, $value = null): array
    {
        if ([] === $values) {
            return [];
        }

        if (null !== $value || isset($this->choices) || null !== $this->queryBuilder) {
            return $this->loadChoiceList($value)->getChoicesForValues($values);
        }

        return $this->findByIdsMaintainingOrder(
            $this->class,
            $values,
            $this->languageId,
            $this->shopId
        );
    }

    /**
     * @return T[]
     */
    private function doLoadChoices(): array
    {
        if (null === $this->queryBuilder) {
            return $this->manager->getRepository($this->class)->findAll($this->languageId, $this->shopId);
        }

        return $this->queryBuilder
            ->setLanguageId($this->languageId)
            ->build()
            ->getResult();
    }
}
