<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Shipment\Form\Type;

use InPost\International\Api\Shipment\Model\Parcel\ParcelType as ParcelTypeEnum;
use InPost\International\Entity\ParcelTemplate;
use InPost\International\Enum\Form\Type\EnumType;
use InPost\International\ParcelTemplate\ParcelTemplateRepositoryInterface;
use InPost\International\Shipment\DTO\Parcel;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Contracts\Translation\TranslatorInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

final class ParcelType extends AbstractType
{
    /**
     * @var TranslatorInterface
     */
    private $translator;

    /**
     * @var ParcelTemplateRepositoryInterface
     */
    private $templateRepository;

    public function __construct(TranslatorInterface $translator, ParcelTemplateRepositoryInterface $templateRepository)
    {
        $this->translator = $translator;
        $this->templateRepository = $templateRepository;
    }

    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('template', EntityType::class, [
                'class' => ParcelTemplate::class,
                'label' => $this->translator->trans('Parcel template', [], 'Modules.Inpostinternational.Shipment'),
                'mapped' => false,
                'choice_attr' => static function (ParcelTemplate $template): array {
                    return [
                        'data-type' => $template->getType()->value,
                        'data-weight' => json_encode([
                            'weight' => $template->getWeight()->getWeight(),
                            'unit' => $template->getWeight()->getUnit(),
                        ]),
                        'data-dimensions' => json_encode([
                            'length' => $template->getDimensions()->getLength(),
                            'width' => $template->getDimensions()->getWidth(),
                            'height' => $template->getDimensions()->getHeight(),
                            'unit' => $template->getDimensions()->getUnit(),
                        ]),
                    ];
                },
                'data' => $this->templateRepository->getDefault(),
            ])
            ->add('type', EnumType::class, [
                'class' => ParcelTypeEnum::class,
                'label' => $this->translator->trans('Parcel type', [], 'Modules.Inpostinternational.Shipment'),
            ])
            ->add('dimensions', DimensionsType::class, [
                'label' => $this->translator->trans('Dimensions', [], 'Modules.Inpostinternational.Shipment'),
            ])
            ->add('weight', WeightType::class, [
                'label' => $this->translator->trans('Weight', [], 'Admin.Global'),
            ])
            ->add('comment', TextType::class, [
                'required' => false,
                'label' => $this->translator->trans('Customer reference', [], 'Modules.Inpostinternational.Shipment'),
            ]);
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => Parcel::class,
        ]);
    }
}
