<?php
/**
 * Copyright since 2025 InPost S.A.
 *
 * For the full license information, please view the LICENSE file bundled with the module.
 *
 * @author InPost S.A.
 * @copyright since 2025 InPost S.A.
 * @license MIT
 */

declare(strict_types=1);

namespace InPost\International\Shipment;

use InPost\International\Api\Shipment\ShipmentType;
use InPost\International\Enum\StringEnum;
use InPost\International\Translation\TranslatableInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * @method static self CourierPickup()
 * @method static self FromPoint()
 */
final class ShippingMethod extends StringEnum implements TranslatableInterface
{
    private const COURIER_PICKUP = 'courier_pickup';
    private const FROM_POINT = 'point';

    public static function fromShipmentType(ShipmentType $shipmentType): self
    {
        switch ($shipmentType) {
            case ShipmentType::PointToPoint():
            case ShipmentType::PointToAddress():
                return self::FromPoint();
            case ShipmentType::AddressToPoint():
            case ShipmentType::AddressToAddress():
                return self::CourierPickup();
            default:
                throw new \LogicException('Not implemented.');
        }
    }

    /**
     * @return ShipmentType[]
     */
    public function getPossibleShipmentTypes(): array
    {
        switch ($this) {
            case self::CourierPickup():
                return [ShipmentType::AddressToPoint(), ShipmentType::AddressToAddress()];
            case self::FromPoint():
                return [ShipmentType::PointToPoint(), ShipmentType::PointToAddress()];
            default:
                throw new \LogicException('Not implemented.');
        }
    }

    public function trans(TranslatorInterface $translator, string $locale = null): string
    {
        switch ($this) {
            case self::CourierPickup():
                return $translator->trans('Courier pickup', [], 'Modules.Inpostinternational.Shipment', $locale);
            case self::FromPoint():
                return $translator->trans('Parcel Locker or ParcelPoint', [], 'Modules.Inpostinternational.Shipment', $locale);
            default:
                return $this->name;
        }
    }
}
